% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/multi_scale.R
\name{ww_multi_scale}
\alias{ww_multi_scale}
\title{Evaluate metrics at multiple scales of aggregation}
\usage{
ww_multi_scale(
  data = NULL,
  truth,
  estimate,
  metrics = list(yardstick::rmse, yardstick::mae),
  grids = NULL,
  ...,
  na_rm = TRUE,
  aggregation_function = "mean",
  autoexpand_grid = TRUE,
  progress = TRUE
)
}
\arguments{
\item{data}{Either a point geometry \code{sf} object containing the columns
specified by the \code{truth} and \code{estimate} arguments, or \code{NULL} if \code{truth} and
\code{estimate} are \code{SpatRaster} objects.}

\item{truth, estimate}{If \code{data} is an \code{sf} object, the names (optionally
unquoted) for the columns in \code{data} containing the true and predicted values,
respectively. If \code{data} is \code{NULL}, \code{SpatRaster} objects with a single layer
containing the true and predicted values, respectively.}

\item{metrics}{Either a \code{\link[yardstick:metric_set]{yardstick::metric_set()}} object, or a list of
functions which will be used to construct a \code{\link[yardstick:metric_set]{yardstick::metric_set()}} object
specifying the performance metrics to evaluate at each scale.}

\item{grids}{Optionally, a list of pre-computed \code{sf} or \code{sfc} objects
specifying polygon boundaries to use for assessments.}

\item{...}{Arguments passed to \code{\link[sf:st_make_grid]{sf::st_make_grid()}}.
\strong{You almost certainly should provide these arguments as lists.}
For instance, passing \code{n = list(c(1, 2))} will create a single 1x2 grid;
passing \code{n = c(1, 2)} will create a 1x1 grid \emph{and} a 2x2 grid.}

\item{na_rm}{Boolean: Should polygons with NA values be removed before
calculating metrics? Note that this does \emph{not} impact how values are
aggregated to polygons: if you want to remove NA values before aggregating,
provide a function to \code{aggregation_function} which will remove NA values.}

\item{aggregation_function}{The function to use to aggregate predictions and
true values at various scales, by default \code{\link[=mean]{mean()}}. For the \code{sf} method,
you can pass any function which takes a single vector and returns a scalar.
For raster methods, any function accepted by
\code{\link[exactextractr:exact_extract]{exactextractr::exact_extract()}} (note that built-in function names must be
quoted). Note that this function does \emph{not} pay attention to the value of
\code{na_rm}; any NA handling you want to do during aggregation should be handled
by this function directly.}

\item{autoexpand_grid}{Boolean: if \code{data} is in geographic coordinates and
\code{grids} aren't provided, the grids generated by \code{\link[sf:st_make_grid]{sf::st_make_grid()}} may not
contain all observations. If \code{TRUE}, this function will automatically expand
generated grids by a tiny factor to attempt to capture all observations.}

\item{progress}{Boolean: if \code{data} is \code{NULL}, should aggregation via
\code{\link[exactextractr:exact_extract]{exactextractr::exact_extract()}} show a progress bar? Separate progress bars
will be shown for each time \code{truth} and \code{estimate} are aggregated.}
}
\value{
A tibble with six columns: \code{.metric}, with the name
of the metric that the row describes; \code{.estimator}, with the name of the
estimator used, \code{.estimate}, with the output of the metric function;
\code{.grid_args}, with the arguments passed to \code{\link[sf:st_make_grid]{sf::st_make_grid()}} via \code{...}
(if any), \code{.grid}, containing the grids used to aggregate predictions,
as well as the aggregated values of \code{truth} and \code{estimate} as well as the
count of non-NA values for each, and \code{.notes}, which (if \code{data} is an \code{sf}
object) will indicate any observations which were not used in a given
assessment.
}
\description{
Evaluate metrics at multiple scales of aggregation
}
\section{Raster inputs}{


If \code{data} is \code{NULL}, then \code{truth} and \code{estimate} should both be \code{SpatRaster}
objects, as created via \code{\link[terra:rast]{terra::rast()}}. These rasters will then be
aggregated to each grid using \code{\link[exactextractr:exact_extract]{exactextractr::exact_extract()}}.

Grids are calculated using the bounding box of \code{truth}, under the assumption
that you may have extrapolated into regions which do not have matching "true"
values. This function does not check that \code{truth} and \code{estimate} overlap at
all, or that they are at all contained within the grid.
}

\section{Creating grid blocks}{


The grid blocks can be controlled by passing arguments to
\code{\link[sf:st_make_grid]{sf::st_make_grid()}} via \code{...}. Some particularly useful arguments include:
\itemize{
\item \code{cellsize}: Target cellsize, expressed as the "diameter" (shortest
straight-line distance between opposing sides; two times the apothem)
of each block, in map units.
\item \code{n}: The number of grid blocks in the x and y direction (columns, rows).
\item \code{square}: A logical value indicating whether to create square (\code{TRUE}) or
hexagonal (\code{FALSE}) cells.
}

If both \code{cellsize} and \code{n} are provided, then the number of blocks requested
by \code{n} of sizes specified by \code{cellsize} will be returned, likely not
lining up with the bounding box of \code{data}. If only \code{cellsize}
is provided, this function will return as many blocks of size
\code{cellsize} as fit inside the bounding box of \code{data}. If only \code{n} is provided,
then \code{cellsize} will be automatically adjusted to create the requested
number of cells.

This function can be used for geographic or projected coordinate reference
systems and expects 2D data.
}

\examples{
\dontshow{if (rlang::is_installed("modeldata")) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
data(ames, package = "modeldata")
ames_sf <- sf::st_as_sf(ames, coords = c("Longitude", "Latitude"), crs = 4326)
ames_model <- lm(Sale_Price ~ Lot_Area, data = ames_sf)
ames_sf$predictions <- predict(ames_model, ames_sf)

ww_multi_scale(
  ames_sf,
  Sale_Price,
  predictions,
  n = list(
    c(10, 10),
    c(1, 1)
  ),
  square = FALSE
)

# or, mostly equivalently
# (there will be a slight difference due to `autoexpand_grid = TRUE`)
grids <- list(
  sf::st_make_grid(ames_sf, n = c(10, 10), square = FALSE),
  sf::st_make_grid(ames_sf, n = c(1, 1), square = FALSE)
)
ww_multi_scale(ames_sf, Sale_Price, predictions, grids = grids)
\dontshow{\}) # examplesIf}
}
\references{
Riemann, R., Wilson, B. T., Lister, A., and Parks, S. (2010). "An effective
assessment protocol for continuous geospatial datasets of forest
characteristics using USFS Forest Inventory and Analysis (FIA) data."
Remote Sensing of Environment 114(10), pp 2337-2352,
doi: 10.1016/j.rse.2010.05.010 .
}
