% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/vectorFunctions.r
\name{pivec}
\alias{pivec}
\alias{currentvec}
\alias{cumulvec}
\title{Key Probability Vectors of Up-and-Down Designs}
\usage{
pivec(cdf, matfun, ...)

currentvec(cdf, matfun, n, startdose = NULL, marginalize = TRUE, ...)

cumulvec(
  cdf,
  matfun,
  n,
  startdose = NULL,
  proportions = TRUE,
  exclude = 0,
  marginalize = TRUE,
  ...
)
}
\arguments{
\item{cdf}{monotone increasing vector with positive-response probabilities. The number of dose levels \eqn{M} is deduced from vector's length.}

\item{matfun}{The function to calculate the TPM. Depends on the specific design; see \code{\link{bcdmat}}. For all functions except \code{classicmat}, user must provide auxiliary parameters via \code{...}. For the k-in-a-row design, use \code{kmatMarg} for \code{pivec()} and \code{kmatFull} otherwise.}

\item{...}{Arguments passed on to the design's matrix-calculating function.}

\item{n}{For \verb{currentvec, cumulvec}, at what step (= after how many observations) in the experiment would you like the vector calculated?}

\item{startdose}{(for \verb{currentvec, cumulvec}), where does the experiment start? To be given as a dose-level index between 1 and \eqn{M}. If left as \code{NULL} (default), function will assume the equivalent of \emph{"fair die roll"} among all doses. User can also specify your own \eqn{M}-length probability vector.}

\item{marginalize}{logical (for \verb{currentvec, cumulvec} when \code{matfun = kmatFull}) should the returned vector be marginalized over dose levels (\code{TRUE}, default), or should the full set with internal states be returned?}

\item{proportions}{Logical (\code{cumulvec} only) Would you like the results returned as proportions (= a probability vector; \code{TRUE}, default), or as cumulative allocation counts?}

\item{exclude}{Integer (\code{cumulvec} only) Should the cumulative distribution exclude a certain number of initial observations? Default 0.}
}
\value{
A vector of allocation frequencies/probabilities for the doses, summing up to 1. Exception: \code{cumulvec(propotions = FALSE)} returns a vector of expected allocation counts, summing up to \code{n - exclude}.
}
\description{
Dose-allocation probability vectors that quantify the instantaneous, cumulative, and asymptotic behavior of Up-and-Down designs.
}
\details{
Up-and-Down designs (UDDs) generate random walk behavior, which concentrates doses around the target quantile. Asymptotically, dose allocations follow a stationary distribution \eqn{\boldsymbol{\pi}} which can be calculated given the number of doses \eqn{M}, and the value of the cdf \eqn{F} at each dose (i.e., the positive-response probabilities), and the specific UDD rules. No matter the starting dose, the allocation distribution converges to \eqn{\boldsymbol{\pi}} at a geometric rate (Diaconis and Stroock, 1991).

Three functions are offered:
\itemize{
\item \code{pivec()} returns \eqn{\boldsymbol{\pi}}.
\item \code{currentvec()} returns the current (instantaneous) allocation distribution at step \code{n}, using the formula from Diaconis and Stroock (1991).
\item \code{cumulvec()} returns the \emph{cumulative} allocations, i.e., the expected proportions (or counts) of allocations during the experiment after \code{n} observations. This function is perhaps of greatest practical use.
}

All functions first calculate the transition probability matrix (TPM), by calling one of the functions described under \code{\link{bcdmat}}. See that help page for more details.
}
\note{
When using the k-in-a-row design, set \code{matfun = kmatMarg} if using \code{pivec}, and otherwise \code{kmatFull}.

At present, these functions are unable to incorporate in the calculations the impact of the recommended "fast start" stage for k-in-a-row and biased-coin designs. Such a stage begins with a classic UD run, until the first "minority" outcome is encountered (1 for below-median targets and vice versa). Generally such a fast start would make small-sample probability vectors approach the asymptotic distribution more quickly.
}
\examples{
#----- Classical UD Example -----#

# An example used in Oron et al. 2022, Fig. 2.
# It is presented here via the original motivating story:
# "Ketofol"  is a commonly-used anesthesia-inducing mix known to combine its 2 components' 
# beneficial properties, while each component mitigates the other's harmful side-effects. 
# In particular:
#     Propofol reduces blood pressure while ketamine raises it.
# What is *not* known at present, is which mix proportions produce 
# 0 "delta-BP" on average among the population. 

# The classical UD design below administers the mix 0-100\% ketamine in 10\% increments.
#    The design will concentrate doses around the point where half the population 
#    experiences 0 "delta-BP". (the 'zeroPt' parameter in the code)

doses = seq(0, 100, 10)
m=length(doses) # 11 dose levels

zeroPt=63 # the zero-BP point, in units of percent ketamine

# We assume a Normal ("Probit") dose-response curve,
#   and calculate the value of F (i.e.,  prob (delta BP > 0) at the doses:
equivF = pnorm( (doses - zeroPt) / 20)
round(equivF, 3)

# The vector below represents the values feeding into the Fig. 2B barplot.
# "startdose = 6" means the experiment begins from the 6th out of 11 doses, i.e., a 50:50 mix.


round(cumulvec(cdf = equivF, matfun = classicmat, startdose = 6, n = 30), 3)

# Compare with the *instantaneous* probability distribution to the 30th patient:

round(currentvec(cdf = equivF, matfun = classicmat, startdose = 6, n = 30), 3)
# Classic up-and-down has quasi-periodic behavior with a (quasi-)period of 2. 
# Compare the instantaneous vectors at n=30 and 29:
round(currentvec(cdf = equivF, matfun = classicmat, startdose = 6, n = 29), 3)
# Note the alternating near-zero values. Distributions at even/odd n "communicate"
#    with each other only via the dose boundaries.

# Lastly, the asymptotic/stationary distribution. Notice there is no 'n' argument.

round(pivec(cdf = equivF, matfun = classicmat), 3)

# The cumulative vector at n=30 is not very far from the asymptotic vector. 
# The main difference is that at n=30 there's still a bit more
#    probability weight at the starting dose.
# We can check how much of that extra weight is from the 1st patient, by excluding that data point:

round(cumulvec(cdf = equivF, matfun = classicmat, startdose = 6, n = 30, exclude = 1), 3)


}
\references{
\itemize{
\item Diaconis P, Stroock D. Geometric Bounds for Eigenvalues of Markov Chains. \emph{Ann. Appl. Probab.} 1991;1(1):36-61.
\item Hughes BD. \emph{Random Walks and Random Environments, Vol. 1.} Oxford University Press, 1995.
\item Oron AP, Souter MJ, Flournoy N. Understanding Research Methods: Up-and-down Designs for Dose-finding. \emph{Anesthesiology} 2022; 137:137–50.
}
}
\seealso{
\itemize{
\item \code{\link{bcdmat}} for the functions calculating transition probability matrices for various up-and-down designs.
\item \code{\link{k2targ}} for target-finding design aids.
}
}
\author{
Assaf P. Oron \code{<assaf.oron.at.gmail.com>}
}
