% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/xmu_make_top_twin_models.R
\name{xmu_make_top_twin_models}
\alias{xmu_make_top_twin_models}
\title{Helper to make a basic top, MZ, and DZ model.}
\usage{
xmu_make_top_twin_models(mzData, dzData, selDVs, sep = NULL, nSib = 2,
  numObsMZ = NULL, numObsDZ = NULL, equateMeans = TRUE,
  type = c("Auto", "FIML", "cov", "cor", "WLS", "DWLS", "ULS"),
  threshType = c("deviationBased"), weightVar = NULL,
  bVector = FALSE, allContinuousMethod = c("cumulants", "marginals"),
  verbose = FALSE)
}
\arguments{
\item{mzData}{Dataframe containing the MZ data}

\item{dzData}{Dataframe containing the DZ data}

\item{selDVs}{List of base (e.g. BMI) (i.e., NOT 'BMI_T1') variable names}

\item{sep}{Used to expand selDVs into selDVs, i.e., "_T" to expand BMI into BMI_T1 and BMI_T2}

\item{nSib}{Number of members per family (default = 2)}

\item{numObsMZ}{Number of MZ observations contributing (for summary data only)}

\item{numObsDZ}{Number of DZ observations contributing (for summary data only)}

\item{equateMeans}{Whether to equate T1 and T2 means (default = TRUE).}

\item{type}{One of 'Auto','FIML','cov', 'cor', 'WLS','DWLS', or 'ULS'. Auto reacts to the incoming mxData type (raw/cov, WLS).
FIML requires that the data are continuous. Remaining options are weighted, diagonally weighted, or unweighted least squares, respectively)}

\item{threshType}{what type of thresholds to implement if needed.}

\item{weightVar}{If provided, a vector objective will be used to weight the data. (default = NULL).}

\item{bVector}{Whether to compute row-wise likelihoods (defaults to FALSE).}

\item{allContinuousMethod}{passed to xmu_make_mxData (for WLS data) c("cumulants", "marginals")}

\item{verbose}{(default = FALSE)}
}
\value{
\itemize{
\item \code{\link{mxModel}}s for top, MZ and DZ.
}
}
\description{
\code{xmu_make_top_twin_models} makes basic \code{top}, \code{MZ}, and \code{DZ} models. It includes thresholds matrices in the twin models if needed.

This is used in  \code{\link[=umxCP]{umxCP()}}, and \code{\link[=umxACE]{umxACE()}} and \code{\link[=umxACEv]{umxACEv()}} and will be added to the other models: \code{\link[=umxGxE]{umxGxE()}}, \code{\link[=umxIP]{umxIP()}},
simplifying code maintenance.

This function takes the \code{mzData} and \code{dzData}, a list of the \code{selDVs}  to analyse (as well as \code{sep} and \code{nSib}), along with other
relevant information such as whether the user wants to equateMeans, and what threshType to use (currently "deviationBased").
It can also handle a weightVar.

\code{varStarts} is computed as \code{sqrt(variance)/3} of the DVs and \code{meanStarts} as the variable means.
For raw data, a check is made for ordered variables. For Binary variables, means are fixed at 0 and
total variance (A+C+E) is fixed at 1. For ordinal variables, the first 2 thresholds are fixed.

\strong{Modeling}

\emph{top model}

For raw and WLS data, \code{top} contains a means matrix. For summary data, the top model contains only a name.

For ordinal data, \code{top} gains \code{top.threshMat} (from a call to \url{umxThresholdMatrix}]). \code{MZ} and \code{DZ} are as with continuous, but adding thresholds.

\emph{MZ and DZ models}

\code{MZ} and \code{DZ} contain the data, and an expectation referencing \code{top.expCovMZ} and \code{top.expMean}, and, if requested,
referencing \code{vector = bVector}. For WLS these are \url{mxExpectationNormal}]  and \url{mxFitFunctionWLS}].
For continuous raw data, MZ and DZ contain \url{mxExpectationNormal}] and \url{mxFitFunctionML}].

For binary, a constraint and algebras are included to constrain \code{Vtot} (A+C+E) to 1.

If a weightVar is detected, this column is added to  mzWeightMatrix/mzWeightMatrix.

If \code{equateMeans} is \code{TRUE}, then the Twin-2 vars in the mean matrix are equated by label with Twin-1.

\strong{Matrices created}

If needed means matrices are added. Decent starts are guessed from the data.
For continuous raw data, top contains a means matrix "expMean".
For Models with ordinal but no binary variables, top adds an \url{umxThresholdMatrix}].
If binary variables are present, matrices and a constraint to hold A+C+E ==1 are added to top.

If ordinal or binary variables are found in raw data, an \code{mxThreshold} matrix is added to handle these.

If a weight variable is offered up, an \code{mzWeightMatrix} will be added.

\strong{Data handling}

In terms of data handling, \code{xmu_make_top_twin_models} was primarily designed to take data.frames and process these into mxData.
It can also, however, handle cov and mxData input.

It can process data into all the types supported by \code{mxData}.

Raw data input with a target of \code{cov} or \code{cor} type requires the \code{numObsMZ} and \code{numObsDZ} to be set.

Type "WLS", "DWLS", or "ULS" will process raw data into a WLS data using \url{xmu_make_mxData}].

Unused columns are dropped.
If you pass in raw data, you can't request type cov/cor yet. Will work on this if desired.
}
\examples{
# ==============
# = Continuous =
# ==============
library(umx)
data(twinData)
selDVs = c("wt", "ht")
mzData = twinData[twinData$zygosity \%in\%  "MZFF",] 
dzData = twinData[twinData$zygosity \%in\%  "DZFF",]
bits = xmu_make_top_twin_models(mzData= mzData, dzData= dzData, selDVs= selDVs, sep= "", nSib= 2)
names(bits) # "top" "MZ"  "DZ" 

# ============================================
# = Bivariate continuous and ordinal example =
# ============================================
data(twinData)
selDVs = c("wt", "obese")
# Cut BMI column to form ordinal obesity variables
ordDVs          = c("obese1", "obese2")
obesityLevels   = c('normal', 'overweight', 'obese')
cutPoints       = quantile(twinData[, "bmi1"], probs = c(.5, .2), na.rm = TRUE)
twinData$obese1 = cut(twinData$bmi1, breaks = c(-Inf, cutPoints, Inf), labels = obesityLevels) 
twinData$obese2 = cut(twinData$bmi2, breaks = c(-Inf, cutPoints, Inf), labels = obesityLevels) 
# Make the ordinal variables into mxFactors (ensure ordered is TRUE, and require levels)
twinData[, ordDVs] = umxFactor(twinData[, ordDVs])
mzData = twinData[twinData$zygosity \%in\%  "MZFF",] 
dzData = twinData[twinData$zygosity \%in\%  "DZFF",]
bits = xmu_make_top_twin_models(mzData= mzData, dzData= dzData, selDVs= selDVs, sep="", nSib= 2)
names(bits) # "top" "MZ"  "DZ" 

# ==============
# = One binary =
# ==============
data(twinData)
cutPoints       = quantile(twinData[, "bmi1"], probs = .2, na.rm = TRUE)
obesityLevels   = c('normal', 'obese')
twinData$obese1 = cut(twinData$bmi1, breaks = c(-Inf, cutPoints, Inf), labels = obesityLevels) 
twinData$obese2 = cut(twinData$bmi2, breaks = c(-Inf, cutPoints, Inf), labels = obesityLevels) 
ordDVs = c("obese1", "obese2")
twinData[, ordDVs] = umxFactor(twinData[, ordDVs])
selDVs = c("wt", "obese")
mzData = twinData[twinData$zygosity \%in\% "MZFF",]
dzData = twinData[twinData$zygosity \%in\% "DZFF",]
bits = xmu_make_top_twin_models(mzData= mzData, dzData= dzData, selDVs= selDVs, sep= "", nSib= 2)

# ============
# = Cov data =
# ============
data(twinData)
selDVs = c("wt")
mz = cov(twinData[twinData$zygosity \%in\%  "MZFF", tvars(selDVs, sep="")], use = "complete")
dz = cov(twinData[twinData$zygosity \%in\%  "DZFF", tvars(selDVs, sep="")], use = "complete")
bits = xmu_make_top_twin_models(mzData= mzData, dzData= dzData, selDVs= selDVs, sep= "", nSib= 2)
}
\seealso{
Other xmu internal not for end user: \code{\link{umxModel}},
  \code{\link{umx}}, \code{\link{xmuHasSquareBrackets}},
  \code{\link{xmuLabel_MATRIX_Model}},
  \code{\link{xmuLabel_Matrix}},
  \code{\link{xmuLabel_RAM_Model}}, \code{\link{xmuMI}},
  \code{\link{xmuMakeDeviationThresholdsMatrices}},
  \code{\link{xmuMakeOneHeadedPathsFromPathList}},
  \code{\link{xmuMakeTwoHeadedPathsFromPathList}},
  \code{\link{xmuMaxLevels}}, \code{\link{xmuMinLevels}},
  \code{\link{xmuPropagateLabels}},
  \code{\link{xmu_assemble_twin_supermodel}},
  \code{\link{xmu_check_levels_identical}},
  \code{\link{xmu_dot_make_paths}},
  \code{\link{xmu_dot_make_residuals}},
  \code{\link{xmu_make_mxData}},
  \code{\link{xmu_safe_run_summary}},
  \code{\link{xmu_set_sep_from_suffix}},
  \code{\link{xmu_simplex_corner}},
  \code{\link{xmu_start_value_list}},
  \code{\link{xmu_starts}}
}
\concept{xmu internal not for end user}
