% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/build_run_modify.r
\name{umxACE}
\alias{umxACE}
\title{umxACE: Build and run a 2-group Cholesky (uni- or multi-variate)}
\usage{
umxACE(name = "ACE", selDVs, selCovs = NULL, dzData, mzData,
  suffix = NULL, dzAr = 0.5, dzCr = 1, addStd = TRUE, addCI = TRUE,
  numObsDZ = NULL, numObsMZ = NULL, boundDiag = NULL, weightVar = NULL,
  equateMeans = TRUE, bVector = FALSE, thresholds = c("deviationBased",
  "WLS"), autoRun = getOption("umx_auto_run"), sep = NULL)
}
\arguments{
\item{name}{The name of the model (defaults to"ACE")}

\item{selDVs}{The variables to include from the data}

\item{selCovs}{(optional) covariates to include from the data (do not include suffix in names)}

\item{dzData}{The DZ dataframe}

\item{mzData}{The MZ dataframe}

\item{suffix}{The suffix for twin 1 and twin 2, often "_T". If set, simplifies
SelDVs, i.e., just "dep" not c("dep_T1", "dep_T2")}

\item{dzAr}{The DZ genetic correlation (defaults to .5, vary to examine assortative mating)}

\item{dzCr}{The DZ "C" correlation (defaults to 1: set to .25 to make an ADE model)}

\item{addStd}{Whether to add the algebras to compute a std model (defaults to TRUE)}

\item{addCI}{Whether to add intervals to compute CIs (defaults to TRUE)}

\item{numObsDZ}{= Number of DZ twins: Set this if you input covariance data}

\item{numObsMZ}{= Number of MZ twins: Set this if you input covariance data}

\item{boundDiag}{= (optional) numeric lbound for diagonal of the a, c, and e matrices, e.g. 0}

\item{weightVar}{= If provided, a vector objective will be used to weight the data. (default = NULL)}

\item{equateMeans}{Whether to equate the means across twins (defaults to TRUE)}

\item{bVector}{Whether to compute row-wise likelihoods (defaults to FALSE)}

\item{thresholds}{How to implement ordinal thresholds c("deviationBased", "WLS")}

\item{autoRun}{Whether to mxRun the model (default TRUE: the estimated model will be returned)}

\item{sep}{allowed as a synonym for "suffix"}
}
\value{
- \code{\link{mxModel}} of subclass mxModel.ACE
}
\description{
A common task in twin modelling involves using the genetic and environmental differences 
between large numbers of pairs of mono-zygotic (MZ) and di-zygotic (DZ) twins reared together
to model the genetic and environmental structure of one, or, typically, several phenotypes
(measured behaviors). umxACE supports modeling with the ACE Cholesky model, a core model 
in behavior genetics (Cardon and Neale, 1996).
}
\details{
This model decomposes phenotypic variance into Additive genetic,
unique environmental (E) and, optionally, either common or shared-environment (C) or 
non-additive genetic effects (D). Scroll down to details for how to use the function, a figure
and multiple examples.

The Cholesky or lower-triangle decomposition allows a model which is both sure to be 
solvable, and also to account for all the variance (with some restrictions) in the data. 
This model creates as many latent A C and E variables as there are phenotypes, and, moving 
from left to right, decomposes the variance in each component into successively restricted 
factors. The following figure shows how the ACE model appears as a path diagram:

\figure{ACE.png}


\strong{Data Input}
The function flexibly accepts raw data, and also summary covariance data 
(in which case the user must also supple numbers of observations for the two input data sets).

\strong{Ordinal Data}
In an important capability, the model transparently handles ordinal (binary or multi-level
ordered factor data) inputs, and can handle mixtures of continuous, binary, and ordinal
data in any combination. An experimental feature is under development to allow Tobit modelling. 

The function also supports weighting of individual data rows. In this case,
the model is estimated for each row individually, then each row likelihood
is multiplied by its weight, and these weighted likelyhoods summed to form
the model-likelihood, which is to be minimised.
This feature is used in the non-linear GxE model functions.

\strong{Additional features}
The umxACE function supports varying the DZ genetic association (defaulting to .5)
to allow exploring assortative mating effects, as well as varying the DZ \dQuote{C} factor
from 1 (the default for modelling family-level effects shared 100% by twins in a pair),
to .25 to model dominance effects.

\emph{note}: Only one of C or D may be estimated simultaneously. This restriction reflects the lack
of degrees of freedom to simultaneously model C and D with only MZ and DZ twin pairs {ref?}.
}
\examples{

# ============================
# = How heritable is height? =
# ============================
require(umx)
data(twinData) # ?twinData from Australian twins.
# Pick the variables
selDVs = c("ht1", "ht2")
mzData <- twinData[twinData$zygosity \%in\% "MZFF", ]
dzData <- twinData[twinData$zygosity \%in\% "DZFF", ]
m1 = umxACE(selDVs = selDVs, dzData = dzData, mzData = mzData) # -2ll= 9659
umxSummary(m1, std = FALSE) # unstandardized
# tip: with report = "html", umxSummary can print the table to your browser!
plot(m1)

# ========================================================
# = Evidence for dominance ? (DZ correlation set to .25) =
# ========================================================
m2 = umxACE("ADE", selDVs = selDVs, dzData = dzData, mzData = mzData, dzCr = .25)
umxCompare(m2, m1) # ADE is better
umxSummary(m2, comparison = m1) # nb: though this is ADE, columns are labeled ACE

# ==============================
# = Univariate model of weight =
# ==============================

# Things to note:
# 1. This variable has a large variance, but umx picks good starts.
# 2. umxACE can figure out variable names: provide "sep" and "wt" -> "wt1" "wt2"
# 3. umxACE picks the variables it needs from the data.
# 4. You can use boundDiag to lbound a, c, and e at 0 (prevents mirror-solutions).
m1 = umxACE(selDVs = "wt", dzData = dzData, mzData = mzData, sep = "", boundDiag = 0)
# We can modify this model, dropping shared environment, and see a comparison
m2 = umxModify(m1, update = "c_r1c1", comparison = TRUE)
# =====================================
# = Bivariate height and weight model =
# =====================================
data(twinData)
mzData <- twinData[twinData$zygosity \%in\% c("MZFF", "MZMM"),]
dzData <- twinData[twinData$zygosity \%in\% c("DZFF", "DZMM", "DZOS"), ]
mzData <- mzData[1:80,] # quicker run to keep CRAN happy
dzData <- dzData[1:80,]
selDVs = c("ht", "wt") # umx will add suffix (in this case "") + "1" or '2'
m1 = umxACE(selDVs = selDVs, dzData = dzData, mzData = mzData, suffix = '')
umxSummary(m1)

# =========================================================
# = Well done! Now you can make modify twin models in umx =
# =========================================================


# ===================
# = Ordinal example =
# ===================
require(umx)
data(twinData)
# Cut bmi column to form ordinal obesity variables
ordDVs = c("obese1", "obese2")
selDVs = c("obese")
obesityLevels = c('normal', 'overweight', 'obese')
cutPoints <- quantile(twinData[, "bmi1"], probs = c(.5, .2), na.rm = TRUE)
twinData$obese1 <- cut(twinData$bmi1, breaks = c(-Inf, cutPoints, Inf), labels = obesityLevels) 
twinData$obese2 <- cut(twinData$bmi2, breaks = c(-Inf, cutPoints, Inf), labels = obesityLevels) 
# Make the ordinal variables into mxFactors (ensure ordered is TRUE, and require levels)
twinData[, ordDVs] <- mxFactor(twinData[, ordDVs], levels = obesityLevels)
mzData <- twinData[twinData$zyg == 1, umx_paste_names(selDVs, "", 1:2)]
dzData <- twinData[twinData$zyg == 3, umx_paste_names(selDVs, "", 1:2)]
mzData <- mzData[1:80,] # just top 80 pairs to run fast
dzData <- dzData[1:80,]
str(mzData) # make sure mz, dz, and t1 and t2 have the same levels!
m1 = umxACE(selDVs = selDVs, dzData = dzData, mzData = mzData, suffix = '')
umxSummary(m1)

# ============================================
# = Bivariate continuous and ordinal example =
# ============================================
data(twinData)
selDVs = c("wt", "obese")
# Cut bmi column to form ordinal obesity variables
ordDVs = c("obese1", "obese2")
obesityLevels = c('normal', 'overweight', 'obese')
cutPoints <- quantile(twinData[, "bmi1"], probs = c(.5, .2), na.rm = TRUE)
twinData$obese1 <- cut(twinData$bmi1, breaks = c(-Inf, cutPoints, Inf), labels = obesityLevels) 
twinData$obese2 <- cut(twinData$bmi2, breaks = c(-Inf, cutPoints, Inf), labels = obesityLevels) 
# Make the ordinal variables into mxFactors (ensure ordered is TRUE, and require levels)
twinData[, ordDVs] <- mxFactor(twinData[, ordDVs], levels = obesityLevels)
mzData <- twinData[twinData$zyg == 1,] # umxACE can trim out unused variables on its own
dzData <- twinData[twinData$zyg == 3,]
mzData <- mzData[1:80,] # just top 80 so example runs in a couple of secs
dzData <- dzData[1:80,]
m1 = umxACE(selDVs = selDVs, dzData = dzData, mzData = mzData, suffix = '')
plot(m1)

# =======================================
# = Mixed continuous and binary example =
# =======================================
require(umx)
data(twinData)
# Cut to form category of 20\% obese subjects
# and make into mxFactors (ensure ordered is TRUE, and require levels)
cutPoints <- quantile(twinData[, "bmi1"], probs = .2, na.rm = TRUE)
obesityLevels = c('normal', 'obese')
twinData$obese1 <- cut(twinData$bmi1, breaks = c(-Inf, cutPoints, Inf), labels = obesityLevels) 
twinData$obese2 <- cut(twinData$bmi2, breaks = c(-Inf, cutPoints, Inf), labels = obesityLevels) 
ordDVs = c("obese1", "obese2")
twinData[, ordDVs] <- mxFactor(twinData[, ordDVs], levels = obesityLevels)

selDVs = c("wt", "obese")
mzData <- twinData[twinData$zygosity \%in\% "MZFF", umx_paste_names(selDVs, "", 1:2)]
dzData <- twinData[twinData$zygosity \%in\% "DZFF", umx_paste_names(selDVs, "", 1:2)]
\dontrun{
m1 = umxACE(selDVs = selDVs, dzData = dzData, mzData = mzData, suffix = '')
umxSummary(m1)
}

# ===================================
# Example with covariance data only =
# ===================================

require(umx)
data(twinData)
selDVs = c("wt1", "wt2")
mz = cov(twinData[twinData$zyg == 1, selDVs], use = "complete")
dz = cov(twinData[twinData$zyg == 3, selDVs], use = "complete")
m1 = umxACE(selDVs = selDVs, dzData = dz, mzData = mz, numObsDZ=569, numObsMZ=351)
umxSummary(m1)
plot(m1)
}
\references{
- \url{http://www.github.com/tbates/umx}
}
\seealso{
Other Twin Modeling Functions: \code{\link{plot.MxModel}},
  \code{\link{umxACESexLim}}, \code{\link{umxACEcov}},
  \code{\link{umxCF_SexLim}}, \code{\link{umxCP}},
  \code{\link{umxGxE_window}}, \code{\link{umxGxE}},
  \code{\link{umxIP}}, \code{\link{umxPlotACEcov}},
  \code{\link{umxPlotCP}}, \code{\link{umxPlotGxE}},
  \code{\link{umxPlotIP}}, \code{\link{umxSummaryACEcov}},
  \code{\link{umxSummaryACE}}, \code{\link{umxSummaryCP}},
  \code{\link{umxSummaryGxE}}, \code{\link{umxSummaryIP}},
  \code{\link{umx}}
}
