% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rs_textures.R
\name{otbtex_hara}
\alias{otbtex_hara}
\title{OTB wrapper for Haralick's simple, advanced and higher order texture features.}
\usage{
otbtex_hara(
  x,
  texture = "all",
  output_name = "hara",
  path_output = NULL,
  return_raster = FALSE,
  parameters.xyrad = list(c(1, 1)),
  parameters.xyoff = list(c(1, 1)),
  parameters.minmax = c(0, 255),
  parameters.nbbin = 8,
  channel = NULL,
  verbose = FALSE,
  otbLinks = NULL,
  ram = "8192"
)
}
\arguments{
\item{x}{A \code{Raster*} object or a \href{http://www.gdal.org/frmt_gtiff.html}{GeoTiff} containing one or more gray  value bands}

\item{texture}{type of filter "all" for all, alternative one of "simple" "advanced" "higher"}

\item{output_name}{string pattern vor individual naming of the output file(s)}

\item{path_output}{path outut}

\item{return_raster}{boolean if TRUE a raster stack is returned}

\item{parameters.xyrad}{list with the x and y radius in pixel indicating the kernel sizes for which the textures are calculated}

\item{parameters.xyoff}{vector containg the directional offsets. Valid combinations are: list(c(1,1),c(1,0),c(0,1),c(1,-1))}

\item{parameters.minmax}{minimum/maximum gray value which can occur.}

\item{parameters.nbbin}{number of gray level bins (classes)}

\item{channel}{sequence of bands to be processed}

\item{verbose}{switch for system messages default is FALSE}

\item{otbLinks}{list. of OTB tools cli pathes}

\item{ram}{reserved memory in MB}
}
\value{
raster* object
}
\description{
OTB wrapper for calculating Haralick's simple, advanced and higher order texture features on every pixel in each channel of the input image
}
\details{
More information at: \href{https://prism.ucalgary.ca/handle/1880/51900}{texture tutorial}
Keep in mind that:\cr
Homogeneity is correlated with Contrast,  r = -0.80\cr
Homogeneity is correlated with Dissimilarity, r = -0.95\cr
GLCM Variance is correlated with Contrast,  r= 0.89\cr
GLCM Variance is correlated with Dissimilarity,  r= 0.91\cr
GLCM Variance is correlated with Homogeneity,  r= -0.83\cr
Entropy is correlated with ASM,  r= -0.87\cr
GLCM Mean and Correlation are more independent. For the same image, GLCM Mean shows  r< 0.1 with any of the other texture measures demonstrated in this tutorial. GLCM Correlation shows  r<0.5 with any other measure.
for a review of a lot of feature extraction algorithms look at: \href{https://doi.org/10.1117/1.JEI.21.2.023016}{Williams et al, 2012, J. of Electronic Imaging, 21(2), 023016 (2012)}\cr
glcm <-> haralick "mean" <-> "advanced 1", "variance" <-> "advanced 2", "homogeneity" <-> "simple 4", "contrast"<-> "simple 5", "dissimilarity" <-> "advanced 2", "entropy" <-> "simple 2", "second_moment"<-> "simple 4", "correlation" <-> "simple 3"
Furthermore using stats will cover mean and variance while dissimilarity is highly correlated to homogeneity data.
}
\examples{
\dontrun{
# load libraries
require(uavRst)
require(link2GI)
require(listviewer)

setwd(tempdir())

# check if OTB exists
otbLinks <- link2GI::linkOTB()

if (otbLinks$exist) {
data("rgb")
raster::plotRGB(rgb)
fn<-file.path(tempdir(),"rgb.tif")
raster::writeRaster(rgb, 
                    filename=fn,
                    format="GTiff", 
                    overwrite=TRUE)
# get help
cmd<-link2GI::parseOTBFunction(algo = "HaralickTextureExtraction",gili=otbLinks)
listviewer::jsonedit(cmd$help)
                       
# calculate simple Haralick-textures for 3 red, green and blue channel
r <- otbtex_hara(x=file.path(tempdir(),"rgb.tif"), 
                 texture = "simple", 
                 return_raster = TRUE, 
                 otbLinks =  otbLinks)

# visualize all layers
raster::plot(r)
}
}
}
\references{
Haralick, R.M., K. Shanmugam and I. Dinstein. 1973. Textural Features for Image Classification.
IEEE Transactions on Systems, Man and Cybernetics. SMC-3(6):610-620.\cr
\href{https://www.orfeo-toolbox.org/SoftwareGuide}{Orfeo Toolbox Sofware Guide, 2016}\cr
\href{https://www.orfeo-toolbox.org//doxygen/classotb_1_1ScalarImageToTexturesFilter.html}{"simple"}:\cr
computes the following 8 local Haralick textures features: Energy, Entropy, Correlation, Inverse Difference Moment, Inertia, Cluster Shade, Cluster Prominence and Haralick Correlation. They are provided in this exact order in the output image. Thus, this application computes the following Haralick textures over a neighborhood with user defined radius.\cr
To improve the speed of computation, a variant of Grey Level Co-occurrence Matrix(GLCM) called Grey Level Co-occurrence Indexed List (GLCIL) is used. Given below is the mathematical explanation on the computation of each textures. Here \code{g( i,j)} is the frequency of element in the GLCIL whose index is \code{i,j}. GLCIL stores a pair of frequency of two pixels from the given offset and the cell index \code{(i,j)} of the pixel in the neighborhood window. Where each element in GLCIL is a pair of pixel index and it's frequency, \code{g(i,j)} is the frequency value of the pair having index is \code{i,j}.\cr\cr
Energy  \if{html}{\figure{form_Energy.png}{options:alt="Energy"}}\cr
Entropy  \if{html}{\figure{form_entropy1.png}{options:alt="Entropy"}}\cr
Correlation  \if{html}{\figure{form_Correlation.png}{options:alt="Correlation"}}\cr
Inertia (contrast)  \if{html}{\figure{form_Contrast.png}{options:alt="Inertia (Contrast)"}}\cr
Cluster Shade  \if{html}{\figure{form_Cluster_Shade.png}{options:alt="Cluster Shade"}}\cr
Cluster Prominence  \if{html}{\figure{form_Cluster_Prominence.png}{options:alt="Cluster Prominence"}}\cr
Haralick's Correlation  \if{html}{\figure{form_Hara_Cor.png}{options:alt="Haralick's Correlation"}}\cr\cr
\href{https://www.orfeo-toolbox.org//doxygen/classotb_1_1ScalarImageToAdvancedTexturesFilter.html}{"advanced"}:\cr
computes the following 10 texture features: Mean, Variance, Dissimilarity, Sum Average, Sum Variance, Sum Entropy, Difference of Entropies, Difference of Variances, IC1 and IC2. They are provided in this exact order in the output image. The textures are computed over a sliding window with user defined radius. To improve the speed of computation, a variant of Grey Level Co-occurrence Matrix(GLCM) called Grey Level Co-occurrence Indexed List (GLCIL) is used. Given below is the mathematical explanation on the computation of each textures. Here \code{g( i,j)} is the frequency of element in the GLCIL whose index is \code{ i,j}. GLCIL stores a pair of frequency of two pixels from the given offset and the cell index \code{( i,j)} of the pixel in the neighborhood window. (where each element in GLCIL is a pair of pixel index and it's frequency, \code{g( i,j)} is the frequency value of the pair having index is \code{ i,j}.\cr\cr

Mean  \if{html}{\figure{form_mean.png}{options:alt="Mean"}}\cr
Sum of squares: Variance  \if{html}{\figure{form_form_sum_of_squares_variance.png}{options:alt="Sum of squares: Variance"}}\cr
Dissimilarity \if{html}{\figure{form_Dissimilarity.png}{options:alt="Dissimilarity"}}\cr
Sum average \if{html}{\figure{form_Sum_average.png}{options:alt="Sum average"}}\cr
Sum Variance \if{html}{\figure{form_Sum_Variance.png}{options:alt="Sum Variance"}}\cr
Sum Entropy \if{html}{\figure{form_Sum_Entropy.png}{options:alt="Sum Entropy"}}\cr
Difference variance \if{html}{\figure{form_Difference_variance.png}{options:alt="Difference variance"}}\cr
Difference entropy \if{html}{\figure{form_Difference_entropy.png}{options:alt="Difference entropy"}}\cr
Information Measures of Correlation IC1 \if{html}{\figure{form_Information_Measures_of_Correlation_IC1.png}{options:alt="Information Measures of Correlation IC1"}}\cr
Information Measures of Correlation IC2 \if{html}{\figure{form_Information_Measures_of_Correlation_IC2.png}{options:alt="Information Measures of Correlation IC2"}}\cr\cr

\href{https://www.orfeo-toolbox.org//doxygen/classotb_1_1ScalarImageToHigherOrderTexturesFilter.html}{"higher"}: \cr\cr
computes 11 local higher order statistics textures coefficients based on the grey level run-length matrix.
It computes the following Haralick textures over a sliding window with user defined radius: (where p( i,j) is the element in cell  i,j of a normalized Run Length Matrix (n_r) is the total number of runs and n_p is the total number of pixels ):\cr

Short Run Emphasis \if{html}{\figure{form_Short_Run_Emphasis.png}{options:alt="Short_Run_Emphasis"}}\cr
Long Run Emphasis \if{html}{\figure{form_Long_Run_Emphasis.png}{options:alt="Long Run Emphasis"}}\cr
Grey-Level Nonuniformity \if{html}{\figure{form_Grey_Level_Nonuniformity.png}{options:alt="Grey-Level Nonuniformity"}}\cr
Run Length Nonuniformity \if{html}{\figure{form_Run_Length_Nonuniformity.png}{options:alt="Run Length Nonuniformity"}}\cr
Low Grey-Level Run Emphasis \if{html}{\figure{form_Low_Grey_Level_Run_Emphasis.png}{options:alt="Low Grey-Level Run Emphasis"}}\cr
High Grey-Level Run Emphasis \if{html}{\figure{form_High_Grey_Level_Run_Emphasis.png}{options:alt="High Grey-Level Run Emphasis"}}\cr
Short Run Low Grey-Level Emphasis \if{html}{\figure{form_Short_Run_Low_Grey_Level_Emphasis.png}{options:alt="Short Run Low Grey-Level Emphasis"}}\cr
Short Run High Grey-Level Emphasis \if{html}{\figure{form_Short_Run_High_Grey_Level_Emphasis.png}{options:alt="Short Run High Grey-Level Emphasis"}}\cr
Long Run Low Grey-Level Emphasis \if{html}{\figure{form_Long_Run_Low_Grey_Level_Emphasis.png}{options:alt="Long Run Low Grey-Level Emphasis"}}\cr
Long Run High Grey-Level Emphasis \if{html}{\figure{form_Long_Run_High_Grey_Level_Emphasis.png}{options:alt="Long Run High Grey-Level Emphasis"}}\cr
}
\author{
Chris Reudenbach
}
