#' @importFrom vars fevd
#' @export
vars::fevd

#' Forecast Error Variance Decomposition
#' 
#' Use the \code{fevd} function from package \pkg{vars} to compute the forecast
#' error variance decomposition of a VAR(p) or VECM for \code{n.ahead} steps.
#' 
#' The function converts the VAR or VECM computed by package tsDyn into an
#' object of class \sQuote{\code{vec2var}}, on which then the
#' \code{\link[vars]{fevd}} method is applied. For details, see package
#' \pkg{vars}.
#' 
#' @param x Object of class \sQuote{\code{VAR}} generated by
#' \command{lineVar()}, or an object of class \sQuote{\code{VECM}} generated by
#' \command{VECM()}
#' @param n.ahead Integer specifying the number of steps.
#' @param ... Currently not used.
#' @return A list with class attribute \sQuote{\code{varfevd}} of length
#' \code{K} holding the forecast error variances as matrices.
#' @author Bernhard Pfaff
#' @seealso \code{\link{plot}} for the plot method. \code{\link{lineVar}},
#' \code{\link{VECM}} for the models.
#' @references Hamilton, J. (1994), \emph{Time Series Analysis}, Princeton
#' University Press, Princeton.
#' 
#' Lutkepohl, H. (2006), \emph{New Introduction to Multiple Time Series
#' Analysis}, Springer, New York.
#' @keywords regression
#' @examples
#' 
#' data(zeroyld)
#' mod_vecm <- VECM(zeroyld, lag = 2)
#' fevd(mod_vecm, n.ahead = 5)
#' 


#' @export
fevd.nlVar <- function(x, n.ahead=10, ...){
  model <- attr(x, "model")
  if(model=="VECM"){
    LRinc <- x$model.specific$LRinclude
    inc <- x$include
    if(LRinc=="both"|inc=="none"&LRinc=="none") warning("Not guaranted to work with this specification of deterministic terms.")
  }
  fevd(vec2var.tsDyn(x),n.ahead=n.ahead, ...)
}

