% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tsesimp.R
\name{tsesimp}
\alias{tsesimp}
\title{The Simple Two-Stage Estimation (TSE) Method for Treatment
Switching}
\usage{
tsesimp(
  data,
  stratum = "",
  time = "time",
  event = "event",
  treat = "treat",
  censor_time = "censor_time",
  pd = "pd",
  pd_time = "pd_time",
  swtrt = "swtrt",
  swtrt_time = "swtrt_time",
  base_cov = "",
  base2_cov = "",
  aft_dist = "weibull",
  strata_main_effect_only = TRUE,
  recensor = TRUE,
  admin_recensor_only = FALSE,
  swtrt_control_only = TRUE,
  alpha = 0.05,
  ties = "efron",
  offset = 1,
  boot = TRUE,
  n_boot = 1000,
  seed = NA
)
}
\arguments{
\item{data}{The input data frame that contains the following variables:
\itemize{
\item \code{stratum}: The stratum.
\item \code{time}: The survival time for right censored data.
\item \code{event}: The event indicator, 1=event, 0=no event.
\item \code{treat}: The randomized treatment indicator, 1=treatment,
0=control.
\item \code{censor_time}: The administrative censoring time. It should
be provided for all subjects including those who had events.
\item \code{pd}: The disease progression indicator, 1=PD, 0=no PD.
\item \code{pd_time}: The time from randomization to PD.
\item \code{swtrt}: The treatment switch indicator, 1=switch, 0=no switch.
\item \code{swtrt_time}: The time from randomization to treatment switch.
\item \code{base_cov}: The values of baseline covariates (excluding treat).
\item \code{base2_cov}: The values of baseline and secondary baseline
covariates (excluding swtrt).
}}

\item{stratum}{The name(s) of the stratum variable(s) in the input data.}

\item{time}{The name of the time variable in the input data.}

\item{event}{The name of the event variable in the input data.}

\item{treat}{The name of the treatment variable in the input data.}

\item{censor_time}{The name of the censor_time variable in the input data.}

\item{pd}{The name of the pd variable in the input data.}

\item{pd_time}{The name of the pd_time variable in the input data.}

\item{swtrt}{The name of the swtrt variable in the input data.}

\item{swtrt_time}{The name of the swtrt_time variable in the input data.}

\item{base_cov}{The vector of names of base_cov variables (excluding
treat) in the input data for the Cox model.}

\item{base2_cov}{The vector of the names of base2_cov variables
(excluding swtrt) in the input data for the AFT model.}

\item{aft_dist}{The assumed distribution for time to event for the AFT
model. Options include "exponential", "weibull", "loglogistic", and
"lognormal".}

\item{strata_main_effect_only}{Whether to only include the strata main
effects in the AFT model. Defaults to \code{TRUE}, otherwise all
possible strata combinations will be considered in the AFT model.}

\item{recensor}{Whether to apply recensoring to counter-factual
survival times. Defaults to \code{TRUE}.}

\item{admin_recensor_only}{Whether to apply recensoring to administrative
censoring time only. Defaults to \code{FALSE}, in which case,
recensoring will be applied to the actual censoring time for dropouts.}

\item{swtrt_control_only}{Whether treatment switching occurred only in
the control group.}

\item{alpha}{The significance level to calculate confidence intervals.}

\item{ties}{The method for handling ties in the Cox model, either
"breslow" or "efron" (default).}

\item{offset}{The offset to calculate the time to event, PD, and treatment
switch. We can set offset equal to 1 (default), 1/30.4375, or 1/365.25
if the time unit is day, month, or year.}

\item{boot}{Whether to use bootstrap to obtain the confidence
interval for hazard ratio. Defaults to \code{TRUE}.}

\item{n_boot}{The number of bootstrap samples.}

\item{seed}{The seed to reproduce the bootstrap results.
The seed from the environment will be used if left unspecified.}
}
\value{
A list with the following components:
\itemize{
\item \code{psi}: The estimated causal parameter for the control group.
\item \code{psi_CI}: The confidence interval for \code{psi}.
\item \code{psi_CI_type}: The type of confidence interval for \code{psi},
i.e., "AFT model" or "bootstrap".
\item \code{logrank_pvalue}: The two-sided p-value of the log-rank test
based on the treatment policy strategy.
\item \code{cox_pvalue}: The two-sided p-value for treatment effect based on
the Cox model.
\item \code{hr}: The estimated hazard ratio from the Cox model.
\item \code{hr_CI}: The confidence interval for hazard ratio.
\item \code{hr_CI_type}: The type of confidence interval for hazard ratio,
either "model" or "bootstrap".
\item \code{settings}: A list with the following components:
\itemize{
\item \code{aft_dist}: The distribution for time to event for the AFT
model.
\item \code{strata_main_effect_only}: Whether to only include the strata
main effects in the AFT model.
\item \code{recensor}: Whether to apply recensoring to counter-factual
survival times.
\item \code{admin_recensor_only}: Whether to apply recensoring to
administrative censoring time only.
\item \code{swtrt_control_only}: Whether treatment switching occurred
only in the control group.
\item \code{alpha}: The significance level to calculate confidence
intervals.
\item \code{ties}: The method for handling ties in the Cox model.
\item \code{offset}: The offset to calculate the time to event, PD, and
treatment switch.
\item \code{boot}: Whether to use bootstrap to obtain the confidence
interval for hazard ratio.
\item \code{n_boot}: The number of bootstrap samples.
\item \code{seed}: The seed to reproduce the simulation results.
}
\item \code{psi_trt}: The estimated causal parameter for the treatment group
if \code{swtrt_control_only} is \code{FALSE}.
\item \code{psi_trt_CI}: The confidence interval for \code{psi_trt} if
\code{swtrt_control_only} is \code{FALSE}.
\item \code{hr_boots}: The bootstrap hazard ratio estimates if \code{boot} is
\code{TRUE}.
\item \code{psi_boots}: The bootstrap \code{psi} estimates if \code{boot} is
\code{TRUE}.
\item \code{psi_trt_boots}: The bootstrap \code{psi_trt} estimates if
\code{boot} is \code{TRUE} and \code{swtrt_control_only} is
\code{FALSE}.
}
}
\description{
Obtains the causal parameter estimate of the AFT model
and the hazard ratio estimate of the Cox model to account for
treatment switching.
}
\details{
We use the following steps to obtain the hazard ratio estimate
and confidence interval had there been no treatment switching:
\itemize{
\item Use an AFT model to estimate the causal parameter \eqn{psi} based on
the patients in the control group who had disease progression.
\item Derive the counter-factual survival times for control patients
had there been no treatment switching.
\item Fit the Cox model to the observed survival times on the treatment arm
and the counter-factual untreated survival times on the control arm
to obtain the hazard ratio estimate.
\item Use bootstrap to construct the p-value and confidence interval for
hazard ratio.
}
}
\examples{

library(dplyr)

# the eventual survival time
shilong1 <- shilong \%>\%
  arrange(bras.f, id, tstop) \%>\%
  group_by(bras.f, id) \%>\%
  filter(row_number() == n()) \%>\%
  select(-c("ps", "ttc", "tran"))

# the last value of time-dependent covariates before pd
shilong2 <- shilong \%>\%
  filter(pd == 0 | tstart <= dpd) \%>\%
  arrange(bras.f, id, tstop) \%>\%
  group_by(bras.f, id) \%>\%
  filter(row_number() == n()) \%>\%
  select(bras.f, id, ps, ttc, tran)

# combine baseline and time-dependent covariates
shilong3 <- shilong1 \%>\%
  left_join(shilong2, by = c("bras.f", "id"))

# apply the two-stage method
fit <- tsesimp(
  data = shilong3, time = "tstop", event = "event",
  treat = "bras.f", censor_time = "dcut", pd = "pd",
  pd_time = "dpd", swtrt = "co", swtrt_time = "dco",
  base_cov = "",
  base2_cov = c("agerand", "sex.f", "tt_Lnum", "rmh_alea.c",
                "pathway.f", "ps", "ttc", "tran"),
  aft_dist = "weibull", alpha = 0.05,
  recensor = TRUE, swtrt_control_only = FALSE, offset = 1,
  boot = FALSE)

c(fit$hr, fit$hr_CI)

}
\references{
Nicholas R Latimer, KR Abrams, PC Lambert, MK Crowther, AJ Wailoo,
JP Morden, RL Akehurst, and MJ Campbell.
Adjusting for treatment switching in randomised controlled trials - A
simulation study and a simplified two-stage method.
Statistical Methods in Medical Research. 2017;26(2):724-751.
}
\author{
Kaifeng Lu, \email{kaifenglu@gmail.com}
}
