#' @title Read `FreeSurfer` surface and volume files
#' @author Zhengjia Wang
#' @details This function is under FreeSurfer license.
#' 1. Volumes:
#' 3D viewer uses `mri/brain.finalsurfs.mgz` from `FreeSurfer` to show the
#' volume information. `brain.finalsurfs.mgz` results from step 1 to 15 in `FreeSurfer`
#' command `recon-all`, which aligns the original `DICOM` image to `RAS` coordinate
#' system, resamples to volume with \code{256x256x256} voxels (tri-linear by default,
#' check \url{https://mail.nmr.mgh.harvard.edu/pipermail/freesurfer/2018-August/058127.html}
#' for more information).
#'
#' 2. Surface:
#' There are two options for surface files. The first choice is using
#' `std.141` brain generated by `AFNI/SUMA`. This surface file re-calculates vertices
#' from standard 141 space, which averages the "surface" of 141 subjects. If you
#' want to map surface electrodes across different subjects, you might want to
#' consider this case as it's especially designed for surface mapping. However,
#' you'll need `AFNI/SUMA` installed to generate the surface file. The details
#' can be found via \url{https://openwetware.org/wiki/Beauchamp:CorticalSurfaceHCP},
#' and the `AFNI/SUMA` command related is `SurfToSurf`. Please generate the files
#' to `[FREESURFER SUBJECT DIR]/SUMA/`. The file name follows the convention of
#' `std.141.[lr]h.[SURFACE TYPE].[POSTFIX]`, where `lh` means left hemisphere and
#' `rh` means right hemisphere; `SURFACE TYPE` can be `pial`, `white`, `smoothwm`,
#' and `POSTFIX` can be `asc`, `gii`. If multiple files for the same surface
#' type exists, the search order will be `asc > gii`.
#' The other option is to use mesh files directly from `FreeSurfer` output located
#' at `[FREESURFER SUBJECT DIR]/surf`. If you want to use these surface, make sure
#' they are converted to `asc` or `gii` format.
#'
#' 3. Electrode registration and transforms
#' This package provides two ways to map electrodes to standard space. For surface
#' electrodes, if standard 141 brain is provided, then the first option is to snap
#' electrodes to the nearest vertices in subject space. The key is the vertex number
#' matches across different subjects, hence the location of corresponding vertices
#' at template brain are the mapped electrode coordinates.
#' If standard 141 brain is missing, or the electrode type is `stereo EEG`, then
#' the second option is volume mapping. The idea is to map electrodes to `MNI305`
#' brain. The details can be found at \url{https://surfer.nmr.mgh.harvard.edu/fswiki/CoordinateSystems}.
#' To perform volume mapping, we need `FreeSurfer` folder `mri/tranforms`.
#' Currently, only linear `Talairach` transform matrix is supported (located at
#' `talairach.xfm`).
#'
#' 4. Coordinates
#' The 3D viewer in this package uses the center of volume as the origin (0, 0, 0).
#'
#' @param fs_subject_folder character, `FreeSurfer` subject folder, or `RAVE` subject folder
#' @param subject_name character, subject code to display with only letters and digits
#' @param additional_surfaces character array, additional surface types to load, such as `white`, `smoothwm`
#' @param use_cache logical, whether to use cached `json` files or from raw `FreeSurfer` files
#' @param use_141 logical, whether to use standard 141 brain for surface file
#'
#' @examples
#' \donttest{
#' # Please run `download_N27()` if `N27` is not at '~/rave_data/others/three_brain/'
#'
#' # Import from `FreeSurfer` subject folder
#' brain = threeBrain::freesurfer_brain(
#'   fs_subject_folder = '~/rave_data/others/three_brain/N27/', subject_name = 'N27',
#'   additional_surfaces = c('white', 'smoothwm')
#' )
#'
#' # Visualize. Alternatively, you can use brain$plot(...)
#' plot( brain )
#' }
#' @name freesurfer_brain
#' @export
freesurfer_brain <- function(fs_subject_folder, subject_name,
                             additional_surfaces = NULL,
                             use_cache = TRUE, use_141 = TRUE){
  # Naming conventions
  #
  # Volume group:   Volume (YAB)
  # Volume:         brain.finalsurfs (YAB)
  #
  # Surface group:  Surface - pial (YAB)
  # Surface         Standard 141 Right Hemisphere - pial (YAB)
  #
  # Electrode container: Electrodes (YAB)
  # Electrode          : YAB, 1 - NA



  ##### Load volume and surface ####
  # fs_subject_folder = '~/rave_data/data_dir/congruency/YAB/rave/fs/'
  # subject_name = 'YAB'
  mustWork = TRUE


  # Main get mgz loader
  mgz_loader = tryCatch({
    nibabel = reticulate::import('nibabel')
    nibabel$load
  }, error = function(e){
    read_fs_mgh_mgz
  })

  # Find folders
  if( dir.exists(file.path(fs_subject_folder, 'fs')) ){
    path_subject = normalizePath(file.path(fs_subject_folder, 'fs'), mustWork = mustWork)
  }else if( dir.exists(file.path(fs_subject_folder, 'rave', 'fs')) ){
    path_subject = normalizePath(file.path(fs_subject_folder, 'rave', 'fs'), mustWork = mustWork)
  }else{
    path_subject = normalizePath(fs_subject_folder, mustWork = mustWork)
  }
  path_cache = file.path(path_subject, 'RAVE')

  # Find target files
  path_brainfinal = normalizePath(file.path(path_subject, 'mri', 'brain.finalsurfs.mgz'), mustWork = mustWork)
  path_xform = normalizePath(file.path(path_subject, 'mri', 'transforms', 'talairach.xfm'), mustWork = mustWork)
  path_suma = normalizePath(file.path(path_subject, 'SUMA'), mustWork = FALSE)
  path_surf = normalizePath(file.path(path_subject, 'surf'), mustWork = FALSE)

  # Get general information from fs output
  brain_finalsurf = mgz_loader(path_brainfinal)
  # get Norig
  Norig = brain_finalsurf$header$get_vox2ras()
  Torig = brain_finalsurf$header$get_vox2ras_tkr()

  # get talairach tranform
  ss = readLines(path_xform)
  xfm = stringr::str_match(ss, '^([-]{0,1}[0-9.]+) ([-]{0,1}[0-9.]+) ([-]{0,1}[0-9.]+) ([-]{0,1}[0-9.]+)[;]{0,1}$')
  xfm = xfm[!is.na(xfm[,1]), -1, drop = FALSE]
  if( nrow(xfm) >= 3 ){
    xfm = xfm[1:3,1:4]
  }else{
    cat2('Cannot parse file talairach.xfm properly.', level = 'WARNING')
    xfm = cbind(diag(c(1,1,1)), 0)
  }
  xfm = as.numeric(xfm)
  dim(xfm) = c(3,4)
  xfm = rbind(xfm, c(0,0,0,1))



  # Generate brain object to return
  brain = Brain2$new(subject_code = subject_name, xfm = xfm, Norig = Norig, Torig = Torig)
  brain$meta$path = list(
    path_subject = path_subject,
    path_cache = path_cache,
    path_brainfinal = path_brainfinal,
    path_xform = path_xform,
    path_suma = path_suma,
    path_surf = path_surf
  )


  ##### get volume 256x256x256 ####
  dir.create(path_cache, recursive = TRUE, showWarnings = FALSE)

  geom_brain_finalsurfs = NULL
  volume_shape = as.integer(brain_finalsurf$get_shape())
  group_volume = GeomGroup$new(name = sprintf('Volume - brain.finalsurfs (%s)', subject_name))
  group_volume$subject_code = subject_name
  cache_volume = file.path(path_cache, sprintf('%s_brain_finalsurfs.json', subject_name))
  # Read from cache
  if( use_cache && file.exists(cache_volume) ){
    # TODO: Read volume cache
    geom_brain_finalsurfs = DataCubeGeom$new(
      name = sprintf('brain.finalsurfs (%s)', subject_name), value = array(NA, dim = volume_shape),
      dim = volume_shape, half_size = volume_shape / 2, group = group_volume,
      position = c(0,0,0), cache_file = cache_volume)
  }else{
    unlink(cache_volume)
  }

  if(is.null(geom_brain_finalsurfs)){
    volume = fill_blanks(brain_finalsurf$get_data(), niter=2)
    # Re-order the data according to Norig, map voxels to RAS coord - anatomical
    order_index = round((Norig %*% c(1,2,3,0))[1:3])
    volume = aperm(volume, abs(order_index))
    sub = sprintf(c('%d:1', '1:%d')[(sign(order_index) + 3) / 2], dim(volume))
    volume = eval(parse(text = sprintf('volume[%s]', paste(sub, collapse = ','))))

    geom_brain_finalsurfs = DataCubeGeom$new(
      name = sprintf('brain.finalsurfs (%s)', subject_name), value = volume, dim = volume_shape,
      half_size = volume_shape / 2, group = group_volume, position = c(0,0,0),
      cache_file = cache_volume)
    rm(volume)
  }
  geom_brain_finalsurfs$subject_code = subject_name

  geom_brain_finalsurfs = BrainVolume$new(
    subject_code = subject_name, volume_type = 'brain.finalsurfs',
    volume = geom_brain_finalsurfs, position = c(0, 0, 0 ))

  brain$add_volume( volume = geom_brain_finalsurfs )

  #### Read surface files ####
  surface_type = unique(c('pial', additional_surfaces))

  for( surf_t in surface_type ){
    surf_lh = NULL; surf_rh = NULL
    surf_group = GeomGroup$new(name = sprintf('Surface - %s (%s)', surf_t, subject_name),
                               position = c( 0, 0, 0 ))
    surf_group$subject_code = subject_name
    surface = NULL
    loaded = FALSE

    if( use_141 ){
      # Load surface from 141 cache
      cache_lh = file.path(path_cache, sprintf('%s_std_141_lh_%s.json', subject_name, surf_t))
      cache_rh = file.path(path_cache, sprintf('%s_std_141_rh_%s.json', subject_name, surf_t))
      if( use_cache && file.exists(cache_lh) && file.exists(cache_rh) ){
        surf_lh = FreeGeom$new(name = sprintf('Standard 141 Left Hemisphere - %s (%s)', surf_t, subject_name),
                               position = c(0,0,0), cache_file = cache_lh, group = surf_group, layer = 8)
        surf_rh = FreeGeom$new(name = sprintf('Standard 141 Right Hemisphere - %s (%s)', surf_t, subject_name),
                               position = c(0,0,0), cache_file = cache_rh, group = surf_group, layer = 8)
        surface = BrainSurface$new(subject_code = subject_name, surface_type = surf_t, mesh_type = 'std.141',
                                   left_hemisphere = surf_lh, right_hemisphere = surf_rh)
        loaded = TRUE
      }else{
        unlink(cache_lh)
        unlink(cache_rh)
      }
      if( !loaded ){
        # try to locate std.141.lh.xxx in SUMA folder
        surf = load_141_surface( path_suma, surf_t = surf_t, quiet = TRUE )
        if( is.null(surf) ){
          # Cannot find any surface
          use_141 = FALSE
        }else{
          surf_lh = FreeGeom$new(name = sprintf('Standard 141 Left Hemisphere - %s (%s)', surf_t, subject_name),
                                 position = c(0,0,0), vertex = surf$left$vertices, face = surf$left$faces,
                                 cache_file = cache_lh, group = surf_group, layer = 8)
          surf_rh = FreeGeom$new(name = sprintf('Standard 141 Right Hemisphere - %s (%s)', surf_t, subject_name),
                                 position = c(0,0,0), vertex = surf$right$vertices, face = surf$right$faces,
                                 cache_file = cache_rh, group = surf_group, layer = 8)
          surface = BrainSurface$new(subject_code = subject_name, surface_type = surf_t, mesh_type = 'std.141',
                                     left_hemisphere = surf_lh, right_hemisphere = surf_rh)
          loaded = TRUE
        }
        rm(surf)
      }

    }

    if( !use_141 ){
      cache_lh = file.path(path_cache, sprintf('%s_fs_lh_%s.json', subject_name, surf_t))
      cache_rh = file.path(path_cache, sprintf('%s_fs_rh_%s.json', subject_name, surf_t))
      if( use_cache && file.exists(cache_lh) && file.exists(cache_rh) ){
        surf_lh = FreeGeom$new(name = sprintf('FreeSurfer Left Hemisphere - %s (%s)', surf_t, subject_name),
                               position = c(0,0,0), cache_file = cache_lh, group = surf_group, layer = 8)
        surf_rh = FreeGeom$new(name = sprintf('FreeSurfer Right Hemisphere - %s (%s)', surf_t, subject_name),
                               position = c(0,0,0), cache_file = cache_rh, group = surf_group, layer = 8)

        surface = BrainSurface$new(subject_code = subject_name, surface_type = surf_t, mesh_type = 'fs',
                                   left_hemisphere = surf_lh, right_hemisphere = surf_rh)
        loaded = TRUE
      }else{
        unlink(cache_lh)
        unlink(cache_rh)
      }
      if( !loaded ){
        # load from fs
        surf = load_fs_surface( path_surf, surf_t = surf_t, quiet = TRUE )
        surf_lh = FreeGeom$new(name = sprintf('FreeSurfer Left Hemisphere - %s (%s)', surf_t, subject_name),
                               position = c(0,0,0), vertex = surf$left$vertices, face = surf$left$faces,
                               cache_file = cache_lh, group = surf_group, layer = 8)
        surf_rh = FreeGeom$new(name = sprintf('FreeSurfer Right Hemisphere - %s (%s)', surf_t, subject_name),
                               position = c(0,0,0), vertex = surf$right$vertices, face = surf$right$faces,
                               cache_file = cache_rh, group = surf_group, layer = 8)
        surface = BrainSurface$new(subject_code = subject_name, surface_type = surf_t, mesh_type = 'fs',
                                   left_hemisphere = surf_lh, right_hemisphere = surf_rh)
        loaded = TRUE
        rm(surf)
      }
    }

    if( 'brain-surface' %in% class(surface) ){

      # This step will automatically adjust position for std.141 mesh
      brain$add_surface( surface = surface )
    }

  }



  surface_type = brain$surface_types
  if( !'pial' %in% surface_type ){
    cat2('Cannot find pial surface. Please make sure fs/SUMA/std.141.[lr]h.pial.asc or fs/surf/[lr]h.pial.asc exists', level = 'FATAL')
  }

  ##### return an environment ####
  return(brain)

}


check_freesurfer_path <- function(fs_subject_folder, autoinstall_template = TRUE){
  if( dir.exists(fs_subject_folder) ){

    if( dir.exists(file.path(fs_subject_folder, 'rave', 'fs')) ){
      path_subject = file.path(fs_subject_folder, 'rave', 'fs')
    }else{
      path_subject = fs_subject_folder
    }
    path_brainfinal = file.path(path_subject, 'mri', 'brain.finalsurfs.mgz')
    path_xform = file.path(path_subject, 'mri', 'transforms', 'talairach.xfm')
    path_surf = file.path(path_subject, 'surf')

    if( file.exists(path_brainfinal) && file.exists(path_xform) && file.exists(path_surf) ){
      return(TRUE)
    }

  }


  # check if this is N27 subject
  subject_code = unlist(stringr::str_split(fs_subject_folder, '/|\\\\'))
  if( subject_code[length(subject_code)] == 'N27' ){
    download_N27()
  }

  return(FALSE)
}

electrode_mapped_141 <- function(position = c(0,0,0), is_surface, vertex_number, surf_type, hemisphere){
  surface_mapping = FALSE

  if( sum(position^2) == 0 ){
    return(TRUE)
  }

  if( isTRUE(is_surface) &&
      length(vertex_number) == 1 && vertex_number >= 0 &&
      length(surf_type) == 1 &&!is.na(surf_type) &&
      length(hemisphere) == 1 && hemisphere %in% c('right', 'left')){
    surface_mapping = TRUE
  }
  if( length(surf_type) == 1 && (is.na(surf_type) || surf_type == 'NA') ){
    # No need to map to surface as there is no surface
    surface_mapping = TRUE
  }
  return(surface_mapping)
}



load_surface_asc_gii_nii <- function(file){

  if( stringr::str_ends(stringr::str_to_lower(file), '\\.asc') ){
    surf = read_fs_asc(file)
    surf$vertices = surf$vertices[,1:3]
    surf$faces = surf$faces[,1:3]
  }else if( stringr::str_ends(stringr::str_to_lower(file), '\\.gii') ){
    # Check if python version is available
    surf = tryCatch({
      # Use python nibabel to load
      nibabel = reticulate::import('nibabel')
      tmp = nibabel$load(file)
      vertices = tmp$darrays[[1]]$data[,1:3]
      faces = tmp$darrays[[2]]$data[,1:3]
      surf = list(
        header = c(nrow(vertices), nrow(faces)),
        vertices = vertices,
        faces = faces
      )
      surf
    }, error = function(e){
      # Use gifti R package to load
      tmp = gifti::readgii(file)
      vertices = tmp$data$pointset[,1:3]
      faces = tmp$data$triangle[,1:3]
      surf = list(
        header = c(nrow(vertices), nrow(faces)),
        vertices = vertices,
        faces = faces
      )
      surf
    })

  }else if( stringr::str_ends(stringr::str_to_lower(file), '\\.nii') ){
    stop('Support for Nifti file has not been implemented')
  }else{
    stop('Only support ASCII, Gifti, Nifti formats. Unknown type')
  }
  return(surf)
}

load_fs_surface <- function(dir, surf_t = 'pial', quiet = FALSE){
  # dir = "/Users/beauchamplab/rave_data/data_dir/congruency/YAB/rave/fs/SUMA"
  lh_file = list.files( dir, pattern = sprintf('^lh\\.%s\\.(asc|gii|nii)', surf_t), full.names = TRUE)
  rh_file = list.files( dir, pattern = sprintf('^rh\\.%s\\.(asc|gii|nii)', surf_t), full.names = TRUE)
  if(any( length(lh_file) == 0, length(rh_file) == 0 )){
    if(!quiet){
      cat2('Cannot find 141 brain from ', dir, level = 'ERROR')
    }
    return(invisible())
  }
  lh_file = sort(lh_file)[1]
  rh_file = sort(rh_file)[1]

  lh_surf = load_surface_asc_gii_nii(lh_file)
  rh_surf = load_surface_asc_gii_nii(rh_file)

  return(list(
    type = surf_t,
    sub_type = 'fs',
    left = lh_surf,
    right = rh_surf
  ))
}



load_141_surface <- function(dir, surf_t = 'pial', quiet = FALSE){
  # dir = "/Users/beauchamplab/rave_data/data_dir/congruency/YAB/rave/fs/SUMA"
  lh_file = list.files( dir, pattern = sprintf('^std\\.141\\.lh\\.%s\\.(asc|gii|nii)', surf_t), full.names = TRUE)
  rh_file = list.files( dir, pattern = sprintf('^std\\.141\\.rh\\.%s\\.(asc|gii|nii)', surf_t), full.names = TRUE)

  if(any( length(lh_file) == 0, length(rh_file) == 0 )){
    if(!quiet){
      cat2('Cannot find 141 brain from ', dir, level = 'ERROR')
    }
    return(invisible())
  }

  lh_file = sort(lh_file)[1]
  rh_file = sort(rh_file)[1]


  lh_surf = load_surface_asc_gii_nii(lh_file)
  rh_surf = load_surface_asc_gii_nii(rh_file)

  return(list(
    type = surf_t,
    sub_type = 'std.141',
    left = lh_surf,
    right = rh_surf
  ))

}
