% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tablespan.R
\name{tablespan}
\alias{tablespan}
\title{tablespan}
\usage{
tablespan(data, formula, title = NULL, subtitle = NULL, footnote = NULL)
}
\arguments{
\item{data}{data set}

\item{formula}{formula to create table}

\item{title}{string specifying the title of the table}

\item{subtitle}{string specifying the subtitle of the table}

\item{footnote}{string specifying the footnote of the table}
}
\value{
Object of class Tablespan with title, subtitle, header info, data, and footnote.
}
\description{
Create complex table spanners with a simple formula.
}
\details{
\code{tablespan} provides a formula based approach to adding headers and spanners
to an existing data.frame. The objective is to provide a unified, easy to use, but good
enough approach to building and exporting tables to Excel, HTML, and LaTeX. To this end,
\code{tablespan} leverages the awesome packages \code{openxlsx} and \code{gt}.

Following the \code{tibble} approach, \code{tablespan} assumes that all items that you may
want to use as row names are just columns in your data set (see example). That
is, \code{tablespan} will allow you to pick some of your items as row names and then just
write them in a separate section to the left of the data.

The table headers are defined with a basic formula approach inspired by \code{tables}.
For example, \code{Species ~ Sepal.Length + Sepal.Width} defines a table with Species as the
row names and Sepal.Length and Sepal.Width as columns. The output will
be similar to the following:
\preformatted{
|Species | Sepal.Length  Sepal.Width|
|:-------|------------: -----------:|
|setosa  |          5.1          3.5|
|setosa  |          4.9          3.0|}

Note that the row names (Species) are in a separate block to the left.

You can add spanner labels with as follows:

\code{Species ~ (Sepal = Sepal.Length + Sepal.Width) + (Petal = Sepal.Length + Sepal.Width)}

This will result in an output similar to:
\preformatted{
|        |           Sepal          |          Petal           |
|Species | Sepal.Length| Sepal.Width| Petal.Length| Petal.Width|
|:-------|------------:|-----------:|------------:|-----------:|
|setosa  |          5.1|         3.5|          1.4|         0.2|}

You can also nest spanners (e.g., \code{Species ~ (Sepal = (Length = Sepal.Length) + (Width = Sepal.Width))}.

When exporting tables, you may want to rename some of you columns. For example,
you may want to rename Sepal.Length and Petal.Length to Length and Sepal.Width and
Petal.Width to Width. With \code{tablespan}, you can rename the item in the header
using \code{new_name:old_name}.
For example, \code{Species ~ (Sepal = Length:Sepal.Length + Width:Sepal.Width) + (Petal = Length:Sepal.Length + Width:Sepal.Width)}
defines a table similar to the following:
\preformatted{
|        |      Sepal     |      Petal     |
|Species | Length | Width | Length | Width |
|:-------|-------:|------:|-------:|------:|
|setosa  |     5.1|    3.5|     1.4|    0.2|}

Finally, to create a table without row names, use
\code{1 ~ (Sepal = Length:Sepal.Length + Width:Sepal.Width) + (Petal = Length:Sepal.Length + Width:Sepal.Width)}
This defines as table similar to the following:
\preformatted{
|      Sepal     |      Petal     |
| Length | Width | Length | Width |
|-------:|------:|-------:|------:|
|     5.1|    3.5|     1.4|    0.2|}

Tables created with \code{tablespan} can be exported to Excel (using \code{openxlsx}),
HTML (using \code{gt}), LaTeX (using \code{gt}), and RTF (using \code{gt}).

References:

\itemize{
 \item{gt: Iannone R, Cheng J, Schloerke B, Hughes E, Lauer A, Seo J, Brevoort K, Roy O (2024). gt: Easily Create Presentation-Ready Display Tables. R package version 0.11.1.9000, <https://github.com/rstudio/gt>, <https://gt.rstudio.com>.}
 \item{tables: Murdoch D (2024). tables: Formula-Driven Table Generation. R package version 0.9.31, <https://dmurdoch.github.io/tables/>.}
 \item{openxlsx: Schauberger P, Walker A (2023). _openxlsx: Read, Write and Edit xlsx Files_. R package version 4.2.5.2, <https://ycphs.github.io/openxlsx/>.}
}
}
\examples{
library(tablespan)
library(dplyr)
data("mtcars")

# We want to report the following table:
summarized_table <- mtcars |>
  group_by(cyl, vs) |>
  summarise(N = n(),
            mean_hp = mean(hp),
            sd_hp = sd(hp),
            mean_wt = mean(wt),
            sd_wt = sd(wt))

# Create a tablespan:
tbl <- tablespan(data = summarized_table,
                 formula = Cylinder:cyl + Engine:vs ~
                   N +
                   (`Horse Power` = Mean:mean_hp + SD:sd_hp) +
                   (`Weight` = Mean:mean_wt + SD:sd_wt),
                 title = "Motor Trend Car Road Tests",
                 subtitle = "A table created with tablespan",
                 footnote = "Data from the infamous mtcars data set.")

tbl

# Export as Excel table:
wb <- as_excel(tbl = tbl)

# Save using openxlsx
# openxlsx::saveWorkbook(wb, "cars.xlsx")

# Export as gt:
gt_tbl <- as_gt(tbl = tbl)
gt_tbl
}
