% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/frame-calendar.R, R/prettify.R
\name{frame_calendar}
\alias{frame_calendar}
\alias{prettify}
\title{Rearrange a temporal data frame to a calendar-based data format using linear algebra}
\usage{
frame_calendar(data, x, y, date, calendar = "monthly", dir = "h",
  week_start = getOption("lubridate.week.start", 1), nrow = NULL,
  ncol = NULL, polar = FALSE, scale = "fixed", width = 0.95,
  height = 0.95, margin = NULL, ...)

prettify(plot, label = c("label", "text"), locale, abbr = TRUE, ...)
}
\arguments{
\item{data}{A data frame or a grouped data frame including a \code{Date} variable.}

\item{x}{A bare (or unquoted) variable mapping to x axis, for example time of
day. If integer 1 is specified, it simply returns calendar grids on x
without transformation.}

\item{y}{A bare (or unquoted) variable or more mapping to y axis. More than
one variable need putting to \code{vars()}. If integer 1 is specified, it returns
calendar grids on y without transformation.}

\item{date}{A \code{Date} variable mapping to dates in the calendar.}

\item{calendar}{Type of calendar. (1) "monthly" calendar (the default) organises
the \code{data} to a common format comprised of day of week in the column and
week of month in the row. A monthly calendar is set up as a 5 by 7 layout
matrix. Each month could extend over six weeks but in these months is to
wrap the last few days up to the top row of the block. (2) "weekly"
calendar consists of day of week and week of year. (3) "daily" calendar
refers to day of month and month of year.}

\item{dir}{Direction of calendar: "h" for horizontal (the default) or "v" for
vertical.}

\item{week_start}{Day on which week starts following ISO conventions -
1 means Monday (default), 7 means Sunday. You can set \code{lubridate.week.start}
option to control this parameter globally.}

\item{nrow, ncol}{Number of rows and columns defined for "monthly" calendar
layout. If \code{NULL}, it computes a sensible layout.}

\item{polar}{FALSE (the default) for Cartesian or TRUE for polar coordinates.}

\item{scale}{"fixed" (the default) for fixed scale. "free" for scaling
conditional on each daily cell, "free_wday" for scaling on weekdays,
"free_mday" for scaling on day of month.}

\item{width, height}{Numerics between 0 and 1 to specify the width/height for
each glyph.}

\item{margin}{Numerics of length two between 0 and 1 to specify the horizontal
and vertical margins between month panels.}

\item{...}{Extra arguments passed to \code{geom_label()} and \code{geom_text()}}

\item{plot}{A "ggplot" object or "plotly".}

\item{label}{If "label" is specified, it will add month/week text on the
\code{ggplot} object, which is actually passed to \code{geom_label()}. If "text" is
specified, it will add weekday/day of month text on the \code{ggplot} object,
which is actually passed to \code{geom_text()}. By default, both "label" and
"text" are used. If "text2" is specified for the "monthly" calendar only,
it will add day of month to the \code{ggplot} object.}

\item{locale}{ISO 639 language code. The default is "en" (i.e. US English).
For other languages support, package \strong{readr} needs to be installed.
See \link[readr:locale]{readr::locale} for more details.}

\item{abbr}{Logical to specify if the abbreviated version of label should be
used.}
}
\value{
A data frame or a dplyr::tibble with newly added columns of \code{.x}, \code{.y}. \code{.x}
and \code{.y} together give new coordinates computed for different types of
calendars. \code{date} groups the same dates in a chronological order, which is
useful for \code{geom_line} or \code{geom_path}. The basic use is \code{ggplot(aes(x = .x, y = .y, group = date)) + geom_*}. The variable names \code{.x} and \code{.y} reflect
the actual \code{x} and \code{y} with a prefix \code{.}.
}
\description{
Temporal data of daily intervals or higher frequency levels can be organised
into a calendar-based format, which is useful for visually presenting
calendar-related activities or multiple seasonality (such as time of day,
day of week, day of month). The function only returns a rearranged data frame,
and \code{ggplot2} takes care of the plotting afterwards. It allows more
flexibility for users to visualise the data in various ways.
}
\details{
The calendar-based graphic can be considered as small multiples
of sub-series arranged into many daily cells. For every multiple (or
facet), it requires the \code{x} variable mapped to be time of day and \code{y} to
value. New \code{x} and \code{y} are computed and named with a \code{.} prefixed to variable
according to \code{x} and \code{y} respectively, and get ready for \code{ggplot2} aesthetic
mappings. In conjunction with \code{group_by()}, it allows the grouped variable
to have their individual scales. For more details, see \code{vignette("frame-calendar", package = "sugrrants")}
}
\examples{
library(dplyr, warn.conflicts = FALSE)
# compute the calendar layout for the data frame
calendar_df <- pedestrian \%>\%
  filter(Sensor_ID == 13, Year == 2016) \%>\%
  frame_calendar(x = Time, y = Hourly_Counts, date = Date, nrow = 4)

# ggplot
p1 <- calendar_df \%>\%
  ggplot(aes(x = .Time, y = .Hourly_Counts, group = Date)) +
  geom_line()
prettify(p1, size = 3, label.padding = unit(0.15, "lines"))

# use in conjunction with group_by()
grped_calendar <- pedestrian \%>\%
  filter(Year == "2017", Month == "March") \%>\%
  group_by(Sensor_Name) \%>\%
  frame_calendar(x = Time, y = Hourly_Counts, date = Date, week_start = 7)

p2 <- grped_calendar \%>\%
  ggplot(aes(x = .Time, y = .Hourly_Counts, group = Date)) +
  geom_line() +
  facet_wrap(~ Sensor_Name, nrow = 2)
prettify(p2)
\dontrun{
# allow for different languages
# below gives simplied Chinese labels with STKaiti font family,
# assuming this font installed in user's local system
prettify(p2, locale = "zh", family = "STKaiti")

# plotly example
if (!requireNamespace("plotly", quietly = TRUE)) {
  stop("Please install the 'plotly' package to run these following examples.")
}
library(plotly)
pp <- calendar_df \%>\% 
  group_by(Date) \%>\%
  plot_ly(x = ~ .Time, y = ~ .Hourly_Counts) \%>\%
  add_lines(text = ~ paste("Count: ", Hourly_Counts, "<br> Time: ", Time))
prettify(pp)
}

}
\seealso{
\link{facet_calendar} for a fully-fledged faceting calendar with formal
labels and axes.
}
