% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/string_tools.R
\name{string_is}
\alias{string_is}
\alias{string_any}
\alias{string_all}
\alias{string_which}
\title{Detects whether a pattern is in a character string}
\usage{
string_is(
  x,
  ...,
  fixed = FALSE,
  ignore.case = FALSE,
  word = FALSE,
  or = FALSE,
  pattern = NULL,
  envir = parent.frame(),
  last = NULL
)

string_any(
  x,
  ...,
  fixed = FALSE,
  ignore.case = FALSE,
  word = FALSE,
  or = FALSE,
  pattern = NULL,
  envir = parent.frame()
)

string_all(
  x,
  ...,
  fixed = FALSE,
  ignore.case = FALSE,
  word = FALSE,
  or = FALSE,
  pattern = NULL,
  envir = parent.frame()
)

string_which(
  x,
  ...,
  fixed = FALSE,
  ignore.case = FALSE,
  word = FALSE,
  or = FALSE,
  pattern = NULL,
  envir = parent.frame()
)
}
\arguments{
\item{x}{A character vector.}

\item{...}{Character scalars representing the patterns to be found. By default they are (perl) regular-expressions.
Use ' & ' or ' | ' to chain patterns and combine their result logically (ex: \code{'[[:alpha:]] & \\\\d'} gets strings
containing both letters and numbers). You can negate by adding a \code{!} first (ex: \code{"!sepal$"} will
return \code{TRUE} for strings that do not end with \code{"sepal"}).
Add flags with the syntax 'flag1, flag2/pattern'. Available flags are: 'fixed', 'ignore', 'word' and 'magic'.
Ex: "ignore/sepal" would get "Sepal.Length" (wouldn't be the case w/t 'ignore').
Shortcut: use the first letters of the flags. Ex: "if/dt[" would get \code{"DT[i = 5]"} (flags 'ignore' + 'fixed').
For 'word', it adds word boundaries to the pattern. The \code{magic} flag first interpolates
values directly into the pattern with "{}".}

\item{fixed}{Logical scalar, default is \code{FALSE}. Whether to trigger a fixed search instead of a
regular expression search (default).}

\item{ignore.case}{Logical scalar, default is \code{FALSE}. If \code{TRUE}, then case insensitive search is triggered.}

\item{word}{Logical scalar, default is \code{FALSE}. If \code{TRUE} then a) word boundaries are added to the pattern,
and b) patterns can be chained by separating them with a comma, they are combined with an OR logical operation.
Example: if \code{word = TRUE}, then pattern = "The, mountain" will select strings containing either the word
'The' or the word 'mountain'.}

\item{or}{Logical, default is \code{FALSE}. In the presence of two or more patterns,
whether to combine them with a logical "or" (the default is to combine them with a logical "and").}

\item{pattern}{(If provided, elements of \code{...} are ignored.) A character vector representing the
patterns to be found. By default a (perl) regular-expression search is triggered.
Use ' & ' or ' | ' to chain patterns and combine their result logically (ex: \code{'[[:alpha:]] & \\\\d'} gets strings
containing both letters and numbers). You can negate by adding a \code{!} first (ex: \code{"!sepal$"} will
return \code{TRUE} for strings that do not end with \code{"sepal"}).
Add flags with the syntax 'flag1, flag2/pattern'. Available flags are: 'fixed', 'ignore', 'word' and 'magic'.
Ex: "ignore/sepal" would get "Sepal.Length" (wouldn't be the case w/t 'ignore').
Shortcut: use the first letters of the flags. Ex: "if/dt[" would get \code{"DT[i = 5]"} (flags 'ignore' + 'fixed').
For 'word', it adds word boundaries to the pattern. The \code{magic} flag first interpolates
values directly into the pattern with "{}".}

\item{envir}{Environment in which to evaluate the interpolations if the flag \code{"magic"} is provided.
Default is \code{parent.frame()}.}

\item{last}{A function or \code{NULL} (default). If a function, it will be applied to the vector
just before returning it.}
}
\value{
It returns a logical vector of the same length as \code{x}.

The function \code{string_which} returns a numeric vector.
}
\description{
Function that detects if one or more patterns are in a string. The patterns can be
chained, by default this is a regex search but special flags be triggered with a
specific syntax, supports negation.
}
\details{
The internal function used to find the patterns is \code{\link[base:grep]{base::grepl()}} with \code{perl = TRUE}.
}
\section{Functions}{
\itemize{
\item \code{string_any()}: Detects if at least one element of a vector matches a regex pattern

\item \code{string_all()}: Detects if all elements of a vector match a regex pattern

\item \code{string_which()}: Returns the indexes of the values in which a pattern is detected

}}
\section{Generic regular expression flags}{


All \code{stringmagic} functions support generic flags in regular-expression patterns.
The flags are useful to quickly give extra instructions, similarly to \emph{usual}
\href{https://javascript.info/regexp-introduction}{regular expression flags}.

Here the syntax is "flag1, flag2/pattern". That is: flags are a comma separated list of flag-names
separated from the pattern with a slash (\code{/}). Example: \code{string_which(c("hello...", "world"), "fixed/.")} returns \code{1}.
Here the flag "fixed" removes the regular expression meaning of "." which would have otherwise meant \emph{"any character"}.
The no-flag verion \code{string_which(c("hello...", "world"), ".")} returns \code{1:2}.

Alternatively, and this is recommended, you can collate the initials of the flags instead of using a
comma separated list. For example: "if/dt[" will apply the flags "ignore" and "fixed" to the pattern "dt[".

The four flags always available are: "ignore", "fixed", "word" and "magic".
\itemize{
\item "ignore" instructs to ignore the case. Technically, it adds the perl-flag "(?i)"
at the beginning of the pattern.
\item "fixed" removes the regular expression interpretation, so that the characters ".", "$", "^", "["
(among others) lose their special meaning and are treated for what they are: simple characters.
\item "word" adds word boundaries (\code{"\\\\b"} in regex language) to the pattern. Further, the comma (\code{","})
becomes a word separator. Technically, "word/one, two" is treated as "\\b(one|two)\\b". Example:
\code{string_clean("Am I ambushed?", "wi/am")} leads to " I ambushed?" thanks to the flags "ignore" and "word".
\item "magic" allows to interpolate variables inside the pattern before regex interpretation.
For example if \code{letters = "aiou"} then \code{string_clean("My great goose!", "magic/[{letters}] => e")}
leads to \code{"My greet geese!"}
}
}

\examples{

# NOTA: using `string_get` instead of `string_is` may lead to a faster understanding 
#       of the examples 

x = string_vec("One, two, one... two, microphone, check")

# default is regular expression search
# => 3 character items
string_is(x, "^...$")

# to trigger fixed search use the flag 'fixed'
string_is(x, "fixed/...")
# you can just use the first letter
string_is(x, "f/...")

# to negate, use '!' as the first element of the pattern
string_is(x, "f/!...")

# you can combine several patterns with " & " or " | "
string_is(x, "one & c")
string_is(x, "one | c")

#
# word: adds word boundaries
#

# compare
string_is(x, "one")
# with
string_is(x, "w/one")

# words can be chained with commas (it is like an OR logical operation)
string_is(x, "w/one, two")
# compare with
string_is(x, "w/one & two")
# remember that you can still negate
string_is(x, "w/one & !two")

# you can combine the flags
# compare
string_is(x, "w/one")
# with
string_is(x, "wi/one")

#
# the `magic` flag
#

p = "one"
string_is(x, "m/{p}")
# Explanation:
# - "p" is interpolated into "one"
# - we get the equivalent: string_is(x, "one")


#
# string_which
#

# it works exactly the same way as string_is
# Which are the items containing an 'e' and an 'o'?
string_which(x, "e", "o")
# equivalently
string_which(x, "e & o")


}
\seealso{
String operations: \code{\link[=string_is]{string_is()}}, \code{\link[=string_get]{string_get()}}, \code{\link[=string_clean]{string_clean()}}, \code{\link[=string_split2df]{string_split2df()}}.
Chain basic operations with \code{\link[=string_ops]{string_ops()}}. Clean character vectors efficiently
with \code{\link[=string_clean]{string_clean()}}.

Use \code{\link[=string_vec]{string_vec()}} to create simple string vectors.

String interpolation combined with operation chaining: \code{\link[=string_magic]{string_magic()}}. You can change \code{string_magic}
default values with \code{\link[=string_magic_alias]{string_magic_alias()}} and add custom operations with \code{\link[=string_magic_register_fun]{string_magic_register_fun()}}.

Display messages while benefiting from \code{string_magic} interpolation with \code{\link[=cat_magic]{cat_magic()}} and \code{\link[=message_magic]{message_magic()}}.

Other tools with aliases: 
\code{\link{cat_magic_alias}()},
\code{\link{string_magic_alias}()},
\code{\link{string_magic}()},
\code{\link{string_ops_alias}()},
\code{\link{string_vec_alias}()}
}
\author{
Laurent R. Berge
}
