#' @title Print the Summary of a 'std_selected' Class Object
#'
#' @description Print the summary generated by [summary()] on the output
#'  of [std_selected()] or [std_selected_boot()].
#'
#' @return
#'  `x` is returned invisibly.
#'
#' @param x The output of [summary()].
#' @param ...  Arguments to be passed to [summary()].
#'
#'
#' @author Shu Fai Cheung <https://orcid.org/0000-0002-9871-9448>
#'
#' @examples
#'
#' # Load a sample data set
#'
#' dat <- test_x_1_w_1_v_1_cat1_n_500
#'
#' # Do a moderated regression by lm
#' lm_raw <- lm(dv ~ iv*mod + v1 + cat1, dat)
#'
#' # Standardize all variables except for categorical variables.
#' # Interaction terms are formed after standardization.
#' lm_std <- std_selected(lm_raw, to_scale = ~ .,
#'                                to_center = ~ .)
#' summary(lm_std)
#'
#' # With bootstrapping
#' # nboot = 100 just for illustration. nboot >= 2000 should be used in read
#' # research.
#' lm_std_boot <- std_selected_boot(lm_raw, to_scale = ~ .,
#'                                          to_center = ~ .,
#'                                          nboot = 100)
#' summary(lm_std_boot)
#'
#' @export

print.summary.std_selected <- function(x, ...) {
  scaled_or_centered <- any(c(!is.null(x$centered_terms), !is.null(x$scaled_terms)))
  if (scaled_or_centered) {
      cat("\nSelected variable(s) are centered by mean and/or scaled by SD")
      if (!is.null(x$centered_terms)) {
          cat("\n- Variable(s) centered:", x$centered_terms)
        }
      if (!is.null(x$scaled_terms)) {
          cat("\n- Variable(s) scaled:", x$scaled_terms)
        }
      cat("\n\n")
    } else {
      cat("\nNo variables are centered by mean or scaled by SD by std_selected().")
      cat("\n\n")
    }
  dat_sc <- format_dat_sc(x)
  print(dat_sc)
  cat("\nNote:")
  cat("\n- Categorical variables will not be centered or scaled even if requested.")
  if (!is.null(x$nboot)) {
      cat("\n- Nonparametric bootstrapping 95% confidence intervals computed.")
      cat("\n- The number of bootstrap samples is ", x$nboot, ".", sep = "")
    }
  cat("\n")
  NextMethod()
  cat("Note:")
  if (scaled_or_centered) {
      cat("\n- Estimates and their statistics are based on the data after",
            "mean-centering, scaling, or standardization.")
    }
  if (!is.null(x$nboot)) {
      cat("\n- [CI Lower, CI Upper] are bootstrap percentile confidence intervals.")
      cat("\n- Std. Error are not bootstrap SEs.")
    }
  cat("\n")
  invisible(x)
}

format_dat_sc <- function(x) {
  dat_sc <- data.frame(centered_by = x$centered_by,
                      scaled_by   = x$scaled_by)
  nonnumeric <- attr(stats::terms(x), "dataClasses") != "numeric"
  dat_sc[nonnumeric, ] <- NA
  centered <- dat_sc$centered_by != 0
  scaled <- dat_sc$scaled_by != 1
  centered[is.na(centered)] <- FALSE
  scaled[is.na(scaled)] <- FALSE
  dat_sc$Note <- ""
  tmpfct <- function(xc, xs) {
      if (xc && xs) {
        return("Standardized (mean = 0, SD = 1)")
      }
      if (xc) return("Centered (mean = 0)")
      if (xs) return("Scaled (SD = 1)")
      return("")
    }
  dat_sc$Note <- format(mapply(tmpfct, centered, scaled))
  dat_sc[is.na(dat_sc$centered_by) &
        is.na(dat_sc$scaled_by), "Note"] <- "Nonnumeric"
  dat_sc$Note <- format(dat_sc$Note)
  dat_sc
}