#' Semi-parametric Analysis of Graph Variability (SP-ANOGVA)
#'
#' \code{sp.anogva} statistically tests whether two or more graphs are
#' generated by the same model and set of parameters.
#'
#' @param Graphs a list of undirected graphs.
#' If each graph has the  attribute \code{eigenvalues} containing its
#' eigenvalues , such values will be used to
#' compute their spectral density.
#'
#' @param model A string that indicates one of the following models: "ER"
#' (Erdos-Renyi random graph model), "GRG" (geometric random graph model), "WS"
#' (Watts-Strogatz random graph model), and "BA" (Barabási-Albert random graph
#' model).
#'
#' @param maxBoot integer indicating the number of bootstrap resamples (default
#' is \code{500}).
#'
#' @param ... Other relevant parameters for \code{\link{graph.param.estimator}}.
#'
#' @return A list with class "htest" containing the following components:
#' \item{\code{statistic:}}{ the F statistic of the test.}
#' \item{\code{p.value:}}{ the p-value of the test.}
#' \item{\code{method:}}{ a string indicating the used method.}
#' \item{\code{data.name:}}{ a string with the data's name(s).}
#' \item{\code{estimates:}}{ a vector containing the estimated parameters for each graph.}
#'
#' @keywords semi_parametric_analysis_of_graph_variability
#'
#' @references
#'
#' Andre Fujita, Eduardo Silva Lira, Suzana de Siqueira Santos, Silvia Yumi
#' Bando, Gabriela Eleuterio Soares, Daniel Yasumasa Takahashi. A
#' semi-parametric statistical test to compare complex networks, Journal of
#' Complex Networks, cnz028, https://doi.org/10.1093/comnet/cnz028
#'
#' Sheather, S. J. and Jones, M. C. (1991). A reliable data-based bandwidth
#' selection method for kernel density estimation.
#' _Journal of the Royal Statistical Society series B_, 53, 683-690.
#' http://www.jstor.org/stable/2345597.
#'
#' @examples
#' \donttest{
#' set.seed(1)
#' model <- "ER"
#' G <- list()
#'
#' # Under H0
#' G[[1]] <- igraph::sample_gnp(50, 0.5)
#' G[[2]] <- igraph::sample_gnp(50, 0.5)
#' G[[3]] <- igraph::sample_gnp(50, 0.5)
#' result1 <- sp.anogva(G, model, maxBoot = 10,eps=0.1)
#' result1
#'
#' # Under H1
#' G[[1]] <- igraph::sample_gnp(50, 0.5)
#' G[[2]] <- igraph::sample_gnp(50, 0.75)
#' G[[3]] <- igraph::sample_gnp(50, 0.5)
#' result2 <- sp.anogva(G, model, maxBoot = 10,eps=0.1)
#' result2
#' }
#'
#'
#' @export
sp.anogva <- function(Graphs, model, maxBoot = 100,...) {
  if(!valid.input(Graphs,level = 1)) stop("The input should be a list of igraph objects!")
  data.name <- deparse(substitute(Graphs))
  # compute and save the spectral density of the graphs
  Graphs <- set.list.spectral.density(Graphs,...)
  nGraphs <- length(Graphs)
  p.hat <- list()
  # estimate the parameter of each graph
  for(l in 1:nGraphs) {
    p.hat[[l]] <- graph.param.estimator(Graph = Graphs[[l]], model = model,...)$param
  }
  #mean_deg = Reduce(f = "+", Map(function (G) { ( igraph::ecount(G)/igraph::vcount(G)) },Graphs))/nGraphs # error
  # obtain the mean degree of all graphs
  mean_deg = Map(function (G) { as.integer(ceiling(igraph::ecount(G)/igraph::vcount(G))) },Graphs)
  # recover the generator functions mantaining the average degree if necessary
  graph_gen_functions = Map(function (mdeg){ get.graph.model(model,mean_deg = mdeg) },mean_deg)

  p.boot <- matrix(0, nGraphs, maxBoot)
  for (l in 1:nGraphs) {
    graph_gen_fun <- graph_gen_functions[[l]]
    n <- igraph::vcount(Graphs[[l]])
	for (boot in 1:maxBoot) {
      Graph <- graph_gen_fun(n,p.hat[[l]])
      p.boot[l,boot] <- graph.param.estimator(Graph = Graph, model = model, ...)$param
      rm(Graph)
    }
  }

  var.boot <- array(0, nGraphs)
  for(l in 1:nGraphs) {
    var.boot[l] <- var(p.boot[l,])
  }
  SSres <- 0
  SStr <- 0
  m <- mean(as.numeric(as.array(p.hat)))
  for(l in 1:nGraphs) {
    SSres <- SSres + (maxBoot-1) * var.boot[l]
    SStr <- SStr + (p.hat[[l]] - m)^2
  }
  F_ <- (SStr / (nGraphs-1)) / ( SSres / (nGraphs*maxBoot - nGraphs))
  p <- pf(F_, df1=(nGraphs-1), df2=(nGraphs*maxBoot - nGraphs), lower.tail=FALSE)
  ###
  names(F_)  <- "F.value"
  estimate <- unlist(p.hat)
  names(estimate) <- paste("parameter", 1:length(estimate), sep='')
  method <- "Semi-parametric Analysis Of Graph Variability"
  rval <- list(statistic=F_, p.value=p, method=method, data.name=data.name, estimate=estimate)
  class(rval) <- "htest"
  return(rval)
}
