% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/optimUSER.R
\name{optimUSER}
\alias{USER}
\alias{optimUSER}
\title{Optimization of sample configurations using a user-defined objective function}
\usage{
optimUSER(points, candi, fun, ..., schedule = scheduleSPSANN(),
  plotit = FALSE, track = FALSE, boundary, progress = "txt",
  verbose = FALSE)
}
\arguments{
\item{points}{Integer value, integer vector, data frame or matrix. If \code{points} is an integer value, 
it defines the number of points that should be randomly sampled from \code{candi} to form the starting 
system configuration. If \code{points} is a vector of integer values, it contains the row indexes of 
\code{candi} that correspond to the points that form the starting system configuration. If \code{points} 
is a data frame or matrix, it must have three columns in the following order: \code{[, "id"]} the row 
indexes of \code{candi} that correspond to each point, \code{[, "x"]} the projected x-coordinates, and 
\code{[, "y"]} the projected y-coordinates. Note that in the later case, \code{points} must be a subset 
of \code{candi}.}

\item{candi}{Data frame or matrix with the candidate locations for the jittered points. \code{candi} must 
have two columns in the following order: \code{[, "x"]} the projected x-coordinates, and \code{[, "y"]} the 
projected y-coordinates.}

\item{fun}{A function defining the objective function that should be used to evaluate the energy state of 
the system configuration at each random perturbation of a candidate sample point. See \sQuote{Details} for 
more information.}

\item{...}{Other arguments passed to the objective function. See \sQuote{Details} for more information.}

\item{schedule}{List with 11 named sub-arguments defining the control parameters of the cooling schedule. 
See \code{\link[spsann]{scheduleSPSANN}}.}

\item{plotit}{Logical for plotting the optimization results, including a) the progress of the objective
function, and b) the starting (gray) and current system configuration (black), and the maximum jitter in 
the x- and y-coordinates. The plots are updated at each 10 jitters. Defaults to \code{plotit = FALSE}.}

\item{track}{Logical value. Should the evolution of the energy state be recorded and returned with the 
result? If \code{track = FALSE} (the default), only the starting and ending energy states are returned with
the results.}

\item{boundary}{SpatialPolygon defining the boundary of the spatial domain. If missing and 
\code{plotit = TRUE}, \code{boundary} is estimated from \code{candi}.}

\item{progress}{Type of progress bar that should be used, with options \code{"txt"}, for a text progress 
bar in the R console, \code{"tk"}, to put up a Tk progress bar widget, and \code{NULL} to omit the progress
bar. A Tk progress bar widget is useful when using parallel processors. Defaults to 
\code{progress = "txt"}.}

\item{verbose}{Logical for printing messages about the progress of the optimization. Defaults to 
\code{verbose = FALSE}.}
}
\value{
\code{optimUSER} returns an object of class \code{OptimizedSampleConfiguration}: the optimized sample
configuration with details about the optimization.
}
\description{
Optimize a sample configuration using a user-defined objective function.
}
\details{
The user-defined objective function \code{fun} must be an object of class \code{\link[base]{function}} and 
include the argument \code{points}. The argument \code{points} is defined in \code{optimUSER} as a matrix 
with three columns: \code{[, 1]} the identification of each sample point given by the respective row 
indexes of \code{candi}, \code{[, 2]} the x-coordinates, and \code{[, 3]} the y-coordinates. The 
identification is useful to retrieve information from any data matrix used by the objective function 
defined by the user.
}
\examples{
\dontrun{
# This example takes more than 5 seconds
require(sp)
require(SpatialTools)
data(meuse.grid)
candi <- meuse.grid[, 1:2]
schedule <- scheduleSPSANN(chains = 1, initial.temperature = 30,
                           x.max = 1540, y.max = 2060, x.min = 0, 
                           y.min = 0, cellsize = 40)

# Define the objective function - number of points per lag distance class
objUSER <-
  function (points, lags, n_lags, n_pts) {
    dm <- SpatialTools::dist1(points[, 2:3])
    ppl <- vector()
    for (i in 1:n_lags) {
      n <- which(dm > lags[i] & dm <= lags[i + 1], arr.ind = TRUE)
      ppl[i] <- length(unique(c(n)))
    }
    distri <- rep(n_pts, n_lags)
    res <- sum(distri - ppl)
  }
lags <- seq(1, 1000, length.out = 10)

# Run the optimization using the user-defined objective function
set.seed(2001)
timeUSER <- Sys.time()
resUSER <- optimUSER(points = 10, fun = objUSER, lags = lags, n_lags = 9,
                     n_pts = 10, candi = candi, schedule = schedule)
timeUSER <- Sys.time() - timeUSER

# Run the optimization using the respective function implemented in spsann
set.seed(2001)
timePPL <- Sys.time()
resPPL <- optimPPL(points = 10, candi = candi, lags = lags, 
                   schedule = schedule)
timePPL <- Sys.time() - timePPL

# Compare results
timeUSER
timePPL
lapply(list(resUSER, resPPL), countPPL, candi = candi, lags = lags)
objSPSANN(resUSER) - objSPSANN(resPPL)
}
}
\author{
Alessandro Samuel-Rosa \email{alessandrosamuelrosa@gmail.com}
}
\concept{
simulated annealing
}
\keyword{iteration}
\keyword{optimize}
\keyword{spatial}

