#' Feature Transformation -- VectorAssembler
#'
#' Combine multiple vectors into a single row-vector; that is,
#' where each row element of the newly generated column is a
#' vector formed by concatenating each row element from the
#' specified input columns.
#'
#' @template roxlate-ml-transformation
#'
#' @export
ft_vector_assembler <- function(x,
                                input_col = NULL,
                                output_col = NULL)
{
  df <- spark_dataframe(x)
  sc <- spark_connection(df)

  assembler <- invoke_new(
    sc,
    "org.apache.spark.ml.feature.VectorAssembler"
  )

  transformed <- assembler %>%
    invoke("setInputCols", as.list(input_col)) %>%
    invoke("setOutputCol", output_col) %>%
    invoke("transform", df)

  sdf_register(transformed)
}

#' Feature Transformation -- StringIndexer
#'
#' Encode a column of labels into a column of label indices.
#' The indices are in [0, numLabels), ordered by label frequencies, with
#' the most frequent label assigned index 0. The transformation
#' can be reversed with \code{\link{ft_index_to_string}}.
#'
#' @template roxlate-ml-transformation
#'
#' @param params An (optional) \R environment -- when available,
#'   the index <-> label mapping generated by the string indexer
#'   will be injected into this environment under the \code{labels}
#'   key.
#'
#' @export
ft_string_indexer <- function(x,
                              input_col = NULL,
                              output_col = NULL,
                              params = NULL)
{
  df <- spark_dataframe(x)
  sc <- spark_connection(df)

  indexer <- invoke_new(
    sc,
    "org.apache.spark.ml.feature.StringIndexer"
  )

  sim <- indexer %>%
    invoke("setInputCol", input_col) %>%
    invoke("setOutputCol", output_col) %>%
    invoke("fit", df)

  # Report labels to caller if requested -- these map
  # the discovered labels in the data set to an associated
  # index.
  if (is.environment(params))
    params$labels <- as.character(invoke(sim, "labels"))

  transformed <- invoke(sim, "transform", df)

  sdf_register(transformed)
}

#' Feature Transformation -- Binarizer
#'
#' Apply thresholding to a column, such that values less than or equal to the
#' \code{threshold} are assigned the value 0.0, and values greater than the
#' threshold are assigned the value 1.0.
#'
#' @template roxlate-ml-transformation
#'
#' @family feature transformation routines
#'
#' @param threshold The numeric threshold.
#'
#' @export
ft_binarizer <- function(x,
                         input_col = NULL,
                         output_col = NULL,
                         threshold = 0.5)
{
  df <- spark_dataframe(x)
  sc <- spark_connection(df)

  binarizer <- invoke_new(
    sc,
    "org.apache.spark.ml.feature.Binarizer"
  )

  transformed <- binarizer %>%
    invoke("setInputCol", input_col) %>%
    invoke("setOutputCol", output_col) %>%
    invoke("setThreshold", as.double(threshold)) %>%
    invoke("transform", df)

  sdf_register(transformed)
}

#' Feature Transformation -- Discrete Cosine Transform (DCT)
#'
#' Transform a column in the time domain into another column in the frequency
#' domain.
#'
#' @template roxlate-ml-transformation
#'
#' @param inverse Perform inverse DCT?
#'
#' @export
ft_discrete_cosine_transform <- function(x,
                                         input_col = NULL,
                                         output_col = NULL,
                                         inverse = FALSE)
{
  df <- spark_dataframe(x)
  sc <- spark_connection(df)

  dct <- invoke_new(
    sc,
    "org.apache.spark.ml.feature.DCT"
  )

  transformed <- dct %>%
    invoke("setInputCol", input_col) %>%
    invoke("setOutputCol", output_col) %>%
    invoke("setInverse", as.logical(inverse)) %>%
    invoke("transform", df)

  sdf_register(transformed)
}

#' Feature Transformation -- IndexToString
#'
#' Symmetrically to \code{\link{ft_string_indexer}},
#' \code{ft_index_to_string} maps a column of label indices back to a
#' column containing the original labels as strings.
#'
#' @template roxlate-ml-transformation
#'
#' @export
ft_index_to_string <- function(x,
                               input_col = NULL,
                               output_col = NULL)
{
  df <- spark_dataframe(x)
  sc <- spark_connection(df)

  converter <- invoke_new(
    sc,
    "org.apache.spark.ml.feature.IndexToString"
  )

  transformed <- converter %>%
    invoke("setInputCol", input_col) %>%
    invoke("setOutputCol", output_col) %>%
    invoke("transform", df)

  sdf_register(transformed)
}

## TODO: These routines with so-called 'row vector' features by
## default, but it would be much nicer to implement routines to
## scale whole columns instead.
# ft_standard_scaler <- function(df, input_col, output_col,
#                                      with.mean, with.std)
# {
#   sc <- spark_connection(df)
#
#   scaler <- invoke_new(
#     sc,
#     "org.apache.spark.ml.feature.StandardScaler"
#   )
#
#   scaler %>%
#     invoke("setInputCol", input_col) %>%
#     invoke("setOutputCol", output_col) %>%
#     invoke("setWithMean", as.logical(with.mean)) %>%
#     invoke("setWithStd", as.logical(with.std)) %>%
#     invoke("transform", df)
# }
#
# ft_min_max_scaler <- function(df, input_col, output_col,
#                                     min = 0, max = 1)
# {
#   sc <- spark_connection(df)
#
#   scaler <- invoke_new(
#     sc,
#     "org.apache.spark.ml.feature.MinMaxScaler"
#   )
#
#   scaler %>%
#     invoke("setInputCol", input_col) %>%
#     invoke("setOutputCol", output_col) %>%
#     invoke("setMin", as.numeric(min)) %>%
#     invoke("setMax", as.numeric(max)) %>%
#     invoke("transform", df)
# }

#' Feature Transformation -- Bucketizer
#'
#' Similar to \R's \code{\link{cut}} function, this transforms a numeric column
#' into a discretized column, with breaks specified through the \code{splits}
#' parameter.
#'
#' @template roxlate-ml-transformation
#'
#' @param splits A numeric vector of cutpoints, indicating the bucket
#'   boundaries.
#'
#' @export
ft_bucketizer <- function(x,
                          input_col = NULL,
                          output_col = NULL,
                          splits)
{
  df <- spark_dataframe(x)
  sc <- spark_connection(df)

  bucketizer <- invoke_new(
    sc,
    "org.apache.spark.ml.feature.Bucketizer"
  )

  transformed <- bucketizer %>%
    invoke("setInputCol", input_col) %>%
    invoke("setOutputCol", output_col) %>%
    invoke("setSplits", as.list(splits)) %>%
    invoke("transform", df)

  sdf_register(transformed)
}

#' Feature Transformation -- ElementwiseProduct
#'
#' Computes the element-wise product between two columns. Generally, this is
#' intended as a scaling transformation, where an input vector is scaled by
#' another vector, but this should apply for all element-wise product
#' transformations.
#'
#' @template roxlate-ml-transformation
#'
#' @param scaling_col The column used to scale \code{input_col}.
#'
#' @export
ft_elementwise_product <- function(x,
                                   input_col = NULL,
                                   output_col = NULL,
                                   scaling_col)
{
  df <- spark_dataframe(x)
  sc <- spark_connection(df)

  transformer <- invoke_new(
    sc,
    "org.apache.spark.ml.feature.ElementwiseProduct"
  )

  transformed <- transformer %>%
    invoke("setInputCol", input_col) %>%
    invoke("setOutputCol", output_col) %>%
    invoke("setScalingVec", scaling_col) %>%
    invoke("transform", df)

  sdf_register(transformed)
}

#' Feature Transformation -- SQLTransformer
#'
#' Transform a data set using SQL. Use the \code{__THIS__}
#' placeholder as a proxy for the active table.
#'
#' @template roxlate-ml-transformation
#'
#' @param sql A SQL statement.
#'
#' @export
ft_sql_transformer <- function(x,
                               input_col = NULL,
                               output_col = NULL,
                               sql)
{
  df <- spark_dataframe(x)
  sc <- spark_connection(df)

  transformer <- invoke_new(
    sc,
    "org.apache.spark.ml.feature.SQLTransformer"
  )

  transformed <- transformer %>%
    invoke("setStatement", paste(sql, collapse = "\n")) %>%
    invoke("transform", df)

  sdf_register(transformed)
}

#' Feature Transformation -- QuantileDiscretizer
#'
#' Takes a column with continuous features and outputs a column with binned
#' categorical features. The bin ranges are chosen by taking a sample of the
#' data and dividing it into roughly equal parts. The lower and upper bin bounds
#' will be -Infinity and +Infinity, covering all real values. This attempts to
#' find numBuckets partitions based on a sample of the given input data, but it
#' may find fewer depending on the data sample values.
#'
#' Note that the result may be different every time you run it, since the sample
#' strategy behind it is non-deterministic.
#'
#' @template roxlate-ml-transformation
#'
#' @param n_buckets The number of buckets to use.
#'
#' @export
ft_quantile_discretizer <- function(x,
                                    input_col = NULL,
                                    output_col = NULL,
                                    n_buckets = 5)
{
  df <- spark_dataframe(x)
  sc <- spark_connection(df)

  discretizer <- invoke_new(
    sc,
    "org.apache.spark.ml.feature.QuantileDiscretizer"
  )

  transformed <- discretizer %>%
    invoke("setInputCol", input_col) %>%
    invoke("setOutputCol", output_col) %>%
    invoke("setNumBuckets", as.numeric(n_buckets)) %>%
    invoke("fit", df) %>%
    invoke("transform", df)

  sdf_register(transformed)
}

#' Feature Transformation -- OneHotEncoder
#'
#' One-hot encoding maps a column of label indices to a column of binary
#' vectors, with at most a single one-value. This encoding allows algorithms
#' which expect continuous features, such as Logistic Regression, to use
#' categorical features.
#'
#' @template roxlate-ml-transformation
#'
#' @export
ft_one_hot_encoder <- function(x, input_col = NULL, output_col = NULL) {

  df <- spark_dataframe(x)
  sc <- spark_connection(df)

  discretizer <- invoke_new(
    sc,
    "org.apache.spark.ml.feature.OneHotEncoder"
  )

  transformed <- discretizer %>%
    invoke("setInputCol", input_col) %>%
    invoke("setOutputCol", output_col) %>%
    invoke("transform", df)

  sdf_register(transformed)
}
