% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sk_plot.R
\name{sk_plot}
\alias{sk_plot}
\title{Plot grid data}
\usage{
sk_plot(g, gdim = NULL, ...)
}
\arguments{
\item{g}{vector, matrix, or any object understood by \code{sk}}

\item{gdim}{numeric vector, (optional) grid dimensions of the data when \code{g} is a vector}

\item{...}{plotting parameters (see details)}
}
\value{
The function returns a vector of suggested plot height and width values in units of
inches which minimize the unused margin space. For example to save a trim version of your
plot as png, call \code{sk_plot} first to get the suggested height and width, say \code{y} and \code{x},
then pass the result to \code{png(filename, height=m*y, width=m*x, pointsize=m*12, ...)},
where \code{m} is any positive scaling factor.
}
\description{
Plots a matrix or raster as a heatmap with an optional color bar legend. This is a wrapper
for \code{graphics::image} similar to \code{terra::plot} but with tighter margins to increase the
image area on screen; and with a different layout for heat-maps of matrices, which are
plotted with i and j axes (row and column numbers) replacing y and x.
}
\details{
This method is called automatically when a "sk" object is passed to \code{plot}.

\code{g} can be a vector, in which case \code{gdim} supplies the y, x dimensions (ie number
of rows, number of columns), in that order. \code{g} can also can be a matrix, raster or any
other object understood by \code{sk} (in which case \code{gdim} can be omitted).

The data in \code{g} can be of numeric, integer, logical, or factor class. The numeric class is
plotted with a continuous color bar legend, while the others get a categorical legend.

Category names (ie tick labels) for the legend can be supplied in argument \code{breaks}, with
one character string for each unique non-NA value in \code{g}, as integer, in ascending order.
If the data are continuous, \code{breaks} can either be the desired number of bins for coloring,
or a vector of break points delineating bins (passed to \code{graphics::image}). Note that a set
of \code{n} bins has \code{n+1} break points.

\code{pal} should be one of the palette names returned by \code{grDevices::hcl.pals}, or else a
vector of color names with length one fewer than the number of break points.

If the data are all \code{NA}, the function omits the heatmap and legend, and draws grid lines
instead. \code{col_grid} can be specified to enable/disable grid lines more generally. These
lines can sometimes appear misaligned due to anti-aliasing. If this is a problem, try
switching to a different graphics device back-end (eg. in Windows Rstudio, try changing from
the default to Cairo with \code{options(RStudioGD.backend = 'cairo')}).

The function sets the graphical parameters 'oma' (to \code{c(0,0,0,0)}) and 'mar' (to values
between 1.1 and 5.1, depending on whether titles, axes, and legend are plotted), then calls
\code{graphics::image}, which sets other graphical parameters such as 'usr'. By default all
graphical parameters are reset to their original values at the end of the function call.
\code{reset=FALSE} prevents this, so that additional elements can be added to the plot later
(such as by calling \code{sf::st_plot(.., add=TRUE)} or \code{graphics:lines}).
}
\section{Plotting parameters}{


The following style parameters are optional:

\describe{

\item{adj, leg_just}{numeric in [0,1]: respectively, the horizontal justification
of the title and vertical justification of the color bar legend (default 0.5 for both) }

\item{asp}{ numeric or NA: the aspect ratio parameter passed to graphics::image (default 1) }

\item{axes, leg}{logical: respectively indicates to draw axes (y and x, or i and j),
the color bar legend (default TRUE)}

\item{axes_w, leg_w, lab_w, main_w, oma_w}{numeric: respectively, the number of lines of
margin to reserve for axes (default 2), legend (default 5), axis labels (default 2), main
title (default 2), and outer white-space (default 0.5)}

\item{breaks}{numeric (vector) or character vector: the color break points (see details)}

\item{cex, cex.main, cex.x, cex.y, cex.z}{numeric: controls the size of text elements in
the plot (default 1), those for title, x/y labels and ticks, and legend title and ticks
all inherit the value assigned to cex (unless otherwise specified).}

\item{col_box, col_grid}{character: respectively, the colors to use for drawing a box
around the image border and for drawing grid cell boundaries (NA to omit)}

\item{col_invert, col_rev}{logical: respectively, inverts (default FALSE), and reverses
the color scale (default TRUE}

\item{ij}{logical: enables/disables matrix style plot with j axis annotations on top
(default TRUE for vector and matrix input, otherwise FALSE)}

\item{layer}{integer: the layer (column) to plot (default 1)}

\item{lwd_axis, lwd_ticks, lwd_grid}{numeric: respectively, line widths for the axis
lines, ticks, and grid lines (default 1)}

\item{main, zlab, ylab, xlab}{character: respectively, a title to put on top in bold,
a legend title to put over the color bar, and axis titles for dimensions y and x.
Setting to '' omits both the label and its margin space}

\item{minimal}{logical: produces a stripped down plot showing only the grid data. This
omits all annotation (unless otherwise specified by \code{axes} and/or \code{leg}) and removes
all margin space (unless otherwise specified by \code{leg_w} and/or \code{mar_w})}

\item{pal}{character (vector): one of \code{graphics::hcl.pals} (default 'Spectral') or a
vector of colors}

\item{pxmax}{integer: the maximum number of pixels to draw along each dimension
(default 2000). If either x or y dimension exceeds this limit, the grid is up-scaled
before plotting}

\item{reset}{logical: indicates to restore original graphical parameters after plot is
finished (default TRUE)}

\item{zlim}{numeric vector: range in the data to plot (ignored for discrete plots)}

\item{x_ontop}{logical: toggles the placement of the horizontal dimension axis on
top vs bottom }

}
}

\examples{
# example grid
gdim = c(50, 100)
n = prod(gdim)
g = sk(gdim)

# plot the grid layout as raster then as matrix
plot(g)
plot(g, ij=TRUE)

# example data: cosine of squared distance to top left corner
z = apply(expand.grid(g$gyx), 1, \(z) cos( 2*sum(z^2) ) )
g_example = utils::modifyList(g, list(gval=z))
plot(g_example)

# plot as matrix (changes default palette)
plot(g_example, ij=TRUE)

# alignment
plot(g_example, ij=TRUE, main='Centered title and legend by default')
plot(g_example, ij=TRUE, main='adj: left-right justification of title', adj=0)
plot(g_example, ij=TRUE, main='leg_just: top-bottom justification of color bar', leg_just=0)

# set the palette - see hcl.pals() for valid names
pal = 'Zissou 1'
plot(g_example, pal=pal, main=pal)
plot(g_example, pal=pal, main=pal, col_invert=TRUE)
plot(g_example, pal=pal, main=pal, col_invert=TRUE, col_rev=TRUE)

# example data: cosine of distance to top left corner
g[] = apply(expand.grid(g$gyx), 1, \(z) cos( sqrt(sum(z^2))/50 ) )
plot(g)

# specify the layer for multi-layer objects (default is first layer)
g_multi = sk(list(gdim=gdim, gval=cbind(z, z^2)))
plot(g_multi)
plot(g_multi, layer=2)

# reduce number of color breaks or specify a factor for discrete value plots
plot(g, breaks=50)
plot(g, breaks=3)
g[] = cut(g[], breaks=3, dig.lab=1)
plot(g)

# pass color bar labels for discrete plots in breaks (in order low to high)
plot(g, breaks=c('a', 'b', 'c'), zlab='group')

# select some "observed" points and make a covariance matrix
idx_obs = match(seq(n), sort(sample.int(prod(gdim), 1e2)))
g[] = idx_obs
plot(g)
v = sk_var(g)

# matrix display mode is automatic when first argument is a matrix or vector
sk_plot(v, zlab=expression(V[ij]))
sk_plot(c(v), dim(v), zlab=expression(V[ij]))

# or pass the matrix to `sk` first to turn it into a sk grid object
g_v = sk(v)
plot(g_v, zlab=expression(V[ij]))

# minimal versions
plot(g_v, minimal=TRUE)
plot(g_v, minimal=TRUE, leg=TRUE)
plot(g_v, minimal=TRUE, col_grid='white', leg=TRUE)

# logical matrix plots are gray-scale by default
plot(g_v > 1e-2, main='logical matrix')

# logical, integer and factor class matrices get a discrete color-bar
interval = 1e-2 # try also 1e-3 to see behaviour with large number of bins
v_discrete = cut(v, seq(0, ceiling(max(v)), by=interval), dig.lab=2)
g_v[] = cut(v, seq(0, ceiling(max(v)), by=interval), dig.lab=2)
plot(g_v)

# labels are preserved for character matrices
z_char = rep(c('foo', 'bar'), n/2)
z_char[sample.int(n, n/2)] = NA
sk_plot(z_char, gdim)

# multi-pane plot
g_sim = sk_sim(c(100, 50), n_layer=3)
split.screen(c(1,3))
screen(1)
plot(g_sim, main='layer 1', layer=1, minimal=TRUE, col_box='black')
screen(2)
plot(g_sim, main='layer 2', layer=2, minimal=TRUE, col_box='black')
screen(3)
plot(g_sim, main='layer 3', layer=3, minimal=TRUE, col_box='black')

}
\seealso{
sk

Other plotting functions: 
\code{\link{sk_plot_pars}()}
}
\concept{plotting functions}
