% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/slim.R
\name{slim}
\alias{slim}
\title{Run a slendr model in SLiM}
\usage{
slim(
  model,
  sequence_length,
  recombination_rate,
  samples = NULL,
  ts = TRUE,
  path = NULL,
  random_seed = NULL,
  method = c("batch", "gui"),
  verbose = FALSE,
  run = TRUE,
  slim_path = NULL,
  burnin = 0,
  max_attempts = 1,
  spatial = !is.null(model$world),
  coalescent_only = TRUE,
  locations = NULL
)
}
\arguments{
\item{model}{Model object created by the \code{compile} function}

\item{sequence_length}{Total length of the simulated sequence (in base-pairs)}

\item{recombination_rate}{Recombination rate of the simulated sequence (in
recombinations per basepair per generation)}

\item{samples}{A data frame of times at which a given number of individuals
should be remembered in the tree-sequence (see \code{schedule_sampling} for a
function that can generate the sampling schedule in the correct format). If
missing, only individuals present at the end of the simulation will be
recorded in the final tree-sequence file.}

\item{ts}{Should a tree sequence be simulated from the model?}

\item{path}{Path to the directory where simulation result files will be saved.
If \code{NULL}, this directory will be automatically created as a temporary
directory. If \code{TRUE}, this path will be also returned by the function.
If a string is given, it is assumed to be a path to a directory where simulation
results will be saved. In this case, the function will return this path invisibly.
Note that if a tree-sequence file should be simulated (along with other files,
potentially), that tree-sequence file (named 'slim.trees' by default) will
have to be explicitly loaded using \code{ts_read()}.}

\item{random_seed}{Random seed (if \code{NULL}, a seed will be generated between
0 and the maximum integer number available)}

\item{method}{How to run the script? ("gui" - open in SLiMgui, "batch" - run
on the command line)}

\item{verbose}{Write the log information from the SLiM run to the console
(default \code{FALSE})?}

\item{run}{Should the SLiM engine be run? If \code{FALSE}, the command line SLiM
command will be printed (and returned invisibly as a character vector) but not executed.}

\item{slim_path}{Path to the appropriate SLiM binary (this is useful if the
\code{slim} binary is not on the \code{$PATH}). Note that this argument must
be specified if the function is being run on Windows.}

\item{burnin}{Length of the burnin (in model's time units, i.e. years)}

\item{max_attempts}{How many attempts should be made to place an offspring
near one of its parents? Serves to prevent infinite loops on the SLiM
backend. Default value is 1.}

\item{spatial}{Should the model be executed in spatial mode? By default, if a
world map was specified during model definition, simulation will proceed in
a spatial mode.}

\item{coalescent_only}{Should \code{initializeTreeSeq(retainCoalescentOnly =
  <...>)} be set to \code{TRUE} (the default) or \code{FALSE}? See
"retainCoalescentOnly" in the SLiM manual for more detail.}

\item{locations}{If \code{NULL}, locations are not saved. Otherwise, the
path to the file where locations of each individual throughout the simulation
will be saved (most likely for use with \code{animate_model}).}
}
\value{
A tree-sequence object loaded via Python-R reticulate interface function \code{ts_read}
(internally represented by the Python object \code{tskit.trees.TreeSequence}). If the
\code{path} argument was set, specifying the directory where results should be saved,
the function will return this path as a single-element character vector.
}
\description{
This function will execute a SLiM script generated by the \code{compile}
function during the compilation of a slendr demographic model.
}
\details{
The arguments \code{sequence_length} and \code{recombination_rate} can be
omitted for slendr models utilizing customized initialization of genomic
architecture. In such cases, users may either provide hard-coded values
directly through SLiM's \code{initializeGenomicElement()} and
\code{initializeRecombinationRate()} functions or utilize slendr's
templating functionality provided by its \code{substitute()} function.
When \code{ts = TRUE}, the returning value of this function depends on whether
or not the \code{path} argument was set. If the user did provide the \code{path}
where output files should be saved, the path is returned (invisibly). This is
mostly intended to support simulations of customized user models. If \code{path}
is not set by the user, it is assumed that a tree-sequence object is desired as
a sole return value of the function (when \code{ts = TRUE}) and so it is automatically
loaded when simulation finishes, or (when \code{ts = FALSE}) that only customized
files are to be produced by the simulation, in which the user will be loading such
files by themselves (and only the path is needed).
}
\examples{
\dontshow{check_dependencies(python = TRUE, slim = TRUE, quit = TRUE) # dependencies must be present
}
init_env()

# load an example model
model <- read_model(path = system.file("extdata/models/introgression", package = "slendr"))

# afr and eur objects would normally be created before slendr model compilation,
# but here we take them out of the model object already compiled for this
# example (in a standard slendr simulation pipeline, this wouldn't be necessary)
afr <- model$populations[["AFR"]]
eur <- model$populations[["EUR"]]
chimp <- model$populations[["CH"]]

# schedule the sampling of a couple of ancient and present-day individuals
# given model at 20 ky, 10 ky, 5ky ago and at present-day (time 0)
modern_samples <- schedule_sampling(model, times = 0, list(afr, 5), list(eur, 5), list(chimp, 1))
ancient_samples <- schedule_sampling(model, times = c(30000, 20000, 10000), list(eur, 1))

# sampling schedules are just data frames and can be merged easily
samples <- rbind(modern_samples, ancient_samples)

# run a simulation using the SLiM back end from a compiled slendr model object and return
# a tree-sequence object as a result
ts <- slim(model, sequence_length = 1e5, recombination_rate = 0, samples = samples)

# simulated tree-sequence object can be saved to a file using ts_write()...
ts_file <- normalizePath(tempfile(fileext = ".trees"), winslash = "/", mustWork = FALSE)
ts_write(ts, ts_file)
# ... and, at a later point, loaded by ts_read()
ts <- ts_read(ts_file, model)

ts
}
