% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/check_model_assumptions.R
\name{check_assumptions}
\alias{check_assumptions}
\alias{outliers}
\alias{heteroskedastic}
\alias{autocorrelation}
\alias{normality}
\alias{multicollin}
\title{Check model assumptions}
\usage{
check_assumptions(x, model.column = NULL, as.logical = FALSE, ...)

outliers(x, iterations = 5)

heteroskedastic(x, model.column = NULL)

autocorrelation(x, model.column = NULL, ...)

normality(x, model.column = NULL)

multicollin(x, model.column = NULL)
}
\arguments{
\item{x}{Fitted \code{lm} (for \code{outliers()}, may also be a \code{glm} model),
or a (nested) data frame with a list-variable that contains fitted model
objects.}

\item{model.column}{Name or index of the list-variable that contains the fitted
model objects. Only applies, if \code{x} is a nested data frame (e.g
with models fitted to \code{\link{bootstrap}} replicates).}

\item{as.logical}{Logical, if \code{TRUE}, the values returned by
\code{check_assumptions()} are \code{TRUE} or \code{FALSE}, indicating
whether each violation of model assumotion holds true or not. If
\code{FALSE} (the default), the p-value of the respective test-statistics
is returned.}

\item{...}{Other arguments, passed down to \code{\link[car]{durbinWatsonTest}}.}

\item{iterations}{Numeric, indicates the number of iterations to remove
outliers.}
}
\value{
A tibble with the respective statistics.
}
\description{
\itemize{
               \item \code{outliers()} detects outliers in (generalized) linear models.
               \item \code{heteroskedastic()} checks a linear model for (non-)constant error variance.
               \item \code{autocorrelation()} checks for independence of errors.
               \item \code{normality()} checks linear models for (non-)normality of residuals.
               \item \code{multicollin()} checks predictors of linear models for multicollinearity.
               \item \code{check_assumptions()} checks all of the above assumptions.
             }
}
\details{
These functions are wrappers that compute various test statistics,
         however, each of them returns a tibble instead of a list of values.
         Furthermore, all functions can also be applied to multiples models
         in stored in \emph{list-variables} (see 'Examples').
         \cr \cr
         \code{outliers()} wraps \code{\link[car]{outlierTest}} and iteratively
         removes outliers for \code{iterations} times, or if the r-squared value
         (for glm: the AIC) did not improve after removing outliers. The function
         returns a tibble with r-squared and AIC statistics for the original
         and updated model, as well as the update model itself (\code{$updated.model}),
         the number (\code{$removed.count}) and indices of the removed observations
         (\code{$removed.obs}).
         \cr \cr
         \code{heteroskedastic()} wraps \code{\link[car]{ncvTest}} and returns
         the p-value of the test statistics as tibble. A p-value < 0.05 indicates
         a non-constant variance (heteroskedasticity).
         \cr \cr
         \code{autocorrelation()} wraps \code{\link[car]{durbinWatsonTest}}
         and returns the p-value of the test statistics as tibble. A p-value
         < 0.05 indicates autocorrelated residuals. In such cases, robust
         standard errors (see \code{\link{robust}} return more accurate results
         for the estimates, or maybe a mixed model with error term for the
         cluster groups should be used.
         \cr \cr
         \code{normality()} calls \code{\link[stats]{shapiro.test}}
         and checks the standardized residuals for normal distribution.
         The p-value of the test statistics is returned as tibble. A p-value
         < 0.05 indicates a significant deviation from normal distribution.
         Note that this formal test almost always yields significant results
         for the distribution of residuals and visual inspection (e.g. qqplots)
         are preferable (see \code{\link[sjPlot]{sjp.lm}} with \code{type = "ma"}).
         \cr \cr
         \code{multicollin()} wraps \code{\link[car]{vif}} and returns
         the logical result as tibble. If \code{TRUE}, multicollinearity
         exists, else not. In case of multicollinearity, the names of independent
         variables that vioalte contribute to multicollinearity are printed
         to the console.
         \cr \cr
         \code{check_assumptions()} runs all of the above tests and returns
         a tibble with all test statistics included. In case the p-values
         are too confusing, use the \code{as.logical} argument, where all
         p-values are replaced with either \code{TRUE} (in case of violation)
         or \code{FALSE} (in case of model conforms to assumption of linar
         regression).
}
\note{
These formal tests are very strict and in most cases violation of model
      assumptions are alerted, thought the model is actually ok. It is
      preferable to check model assumptions based on visual inspection
      (see \code{\link[sjPlot]{sjp.lm}} with \code{type = "ma"}).
}
\examples{
data(efc)

fit <- lm(barthtot ~ c160age + c12hour + c161sex + c172code, data = efc)
outliers(fit)
heteroskedastic(fit)
autocorrelation(fit)
normality(fit)
check_assumptions(fit)

fit <- lm(barthtot ~ c160age + c12hour + c161sex + c172code + neg_c_7,
          data = efc)
outliers(fit)
check_assumptions(fit, as.logical = TRUE)

# apply function to multiple models in list-variable
library(dplyr)
tmp <- efc \%>\%
  bootstrap(50) \%>\%
  mutate(models = lapply(.$strap, function(x) {
    lm(neg_c_7 ~ e42dep + c12hour + c161sex, data = x)
  }))

# for list-variables, argument 'model.column' is the
# quoted name of the list-variable with fitted models
tmp \%>\% heteroskedastic("models")

# Durbin-Watson-Test from package 'car' takes a little bit longer due
# to simulation of p-values...
\dontrun{
tmp \%>\% check_assumptions("models", as.logical = TRUE, reps = 100)}

}
