% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sits_segmentation.R
\name{sits_segment}
\alias{sits_segment}
\title{Segment an image}
\usage{
sits_segment(
  cube,
  seg_fn = sits_slic(),
  roi = NULL,
  impute_fn = impute_linear(),
  start_date = NULL,
  end_date = NULL,
  memsize = 4L,
  multicores = 2L,
  output_dir,
  version = "v1",
  progress = TRUE
)
}
\arguments{
\item{cube}{Regular data cube}

\item{seg_fn}{Function to apply the segmentation}

\item{roi}{Region of interest (see below)}

\item{impute_fn}{Imputation function to remove NA values.}

\item{start_date}{Start date for the segmentation}

\item{end_date}{End date for the segmentation.}

\item{memsize}{Memory available for classification (in GB).}

\item{multicores}{Number of cores to be used for classification.}

\item{output_dir}{Directory for output file.}

\item{version}{Version of the output (for multiple
segmentations).}

\item{progress}{Show progress bar?}
}
\value{
A tibble of class 'segs_cube' representing the
segmentation.
}
\description{
Apply a spatial-temporal segmentation on a data cube based on a user defined
segmentation function. The function applies the segmentation algorithm
"seg_fn" to each tile. The output is a vector data cube, which is a data cube
with an additional vector file in "geopackage" format.
}
\note{
Segmentation requires the following steps:
\enumerate{
 \item Create a regular data cube with \code{\link[sits]{sits_cube}} and
       \code{\link[sits]{sits_regularize}};
 \item Run \code{\link[sits]{sits_segment}} to obtain a vector data cube
       with polygons that define the boundary of the segments;
 \item Classify the time series associated to the segments
       with \code{\link[sits]{sits_classify}}, to get obtain
       a vector probability cube;
 \item Use \code{\link[sits]{sits_label_classification}} to label the
     vector probability cube;
 \item Display the results with \code{\link[sits]{plot}} or
       \code{\link[sits]{sits_view}}.
}
   The "roi" parameter defines a region of interest. It can be
   an sf_object, a shapefile, or a bounding box vector with
   named XY values ("xmin", "xmax", "ymin", "ymax") or
   named lat/long values ("lon_min", "lat_min", "lon_max", "lat_max").

   As of version 1.5.3, the only \code{seg_fn} function available is
   \code{\link[sits]{sits_slic}}, which uses the Simple Linear
   Iterative Clustering (SLIC) algorithm that clusters pixels to
   generate compact, nearly uniform superpixels. This algorithm has been
   adapted by Nowosad and Stepinski to work with multispectral and
   multitemporal images. SLIC uses spectral similarity and
   proximity in the spectral and temporal space to
   segment the image into superpixels. Superpixels are clusters of pixels
   with similar spectral and temporal responses that are spatially close.

   The result of \code{sits_segment} is a data cube tibble with an additional
   vector file in the \code{geopackage} format. The location of the vector
   file is included in the data cube tibble in a new column, called
   \code{vector_info}.
}
\examples{
if (sits_run_examples()) {
    data_dir <- system.file("extdata/raster/mod13q1", package = "sits")
    # create a data cube
    cube <- sits_cube(
        source = "BDC",
        collection = "MOD13Q1-6.1",
        data_dir = data_dir
    )
    # segment the vector cube
    segments <- sits_segment(
        cube = cube,
        seg_fn = sits_slic(
            step = 10,
            compactness = 1,
            dist_fun = "euclidean",
            avg_fun = "median",
            iter = 30,
            minarea = 10
        ),
        output_dir = tempdir()
    )
    # create a classification model
    rfor_model <- sits_train(samples_modis_ndvi, sits_rfor())
    # classify the segments
    seg_probs <- sits_classify(
        data = segments,
        ml_model = rfor_model,
        output_dir = tempdir()
    )
    # label the probability segments
    seg_label <- sits_label_classification(
        cube = seg_probs,
        output_dir = tempdir()
    )
}
}
\references{
Achanta, Radhakrishna, Appu Shaji, Kevin Smith, Aurelien Lucchi,
        Pascal Fua, and Sabine Süsstrunk. 2012. “SLIC Superpixels Compared
        to State-of-the-Art Superpixel Methods.” IEEE Transactions on
        Pattern Analysis and Machine Intelligence 34 (11): 2274–82.

        Nowosad, Jakub, and Tomasz F. Stepinski. 2022. “Extended SLIC
        Superpixels Algorithm for Applications to Non-Imagery Geospatial
        Rasters.” International Journal of Applied Earth Observation
        and Geoinformation 112 (August): 102935.
}
\author{
Gilberto Camara, \email{gilberto.camara@inpe.br}

Rolf Simoes, \email{rolfsimoes@gmail.com}

Felipe Carvalho, \email{felipe.carvalho@inpe.br}

Felipe Carlos, \email{efelipecarlos@gmail.com}
}
