#  Copyright (c) 2024 Merck & Co., Inc., Rahway, NJ, USA and its affiliates.
#  All rights reserved.
#
#  This file is part of the simtrial program.
#
#  simtrial is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.

#' Cut a dataset for analysis at a specified date
#'
#' @param x A time-to-event dataset, for example, generated by [sim_pw_surv()].
#' @param cut_date Date relative to start of randomization
#'   (`cte` from input dataset) at which dataset is to be cut off for analysis.
#'
#' @return A dataset ready for survival analysis.
#'
#' @importFrom data.table ":=" as.data.table setDF
#'
#' @export
#'
#' @examples
#' # Use default enrollment and event rates and
#' # cut at calendar time 5 after start of randomization
#' sim_pw_surv(n = 20) |> cut_data_by_date(5)
cut_data_by_date <- function(x, cut_date) {
  ans <- as.data.table(x)
  ans <- ans[enroll_time <= cut_date, ]
  ans[, tte := pmin(cte, cut_date) - enroll_time]
  ans[, event := fail * (cte <= cut_date)]
  ans <- ans[, c("tte", "event", "stratum", "treatment")]

  setDF(ans)
  return(ans)
}
