% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/boxes.R
\name{box}
\alias{box}
\title{Create a box for the main body of a dashboard}
\usage{
box(..., title = NULL, footer = NULL, status = NULL,
  solidHeader = FALSE, background = NULL, width = 6, height = NULL,
  collapsible = FALSE, collapsed = FALSE)
}
\arguments{
\item{...}{Contents of the box.}

\item{title}{Optional title.}

\item{footer}{Optional footer text.}

\item{status}{The status of the item This determines the item's background
color. Valid statuses are listed in \link{validStatuses}.}

\item{solidHeader}{Should the header be shown with a solid color background?}

\item{background}{If NULL (the default), the background of the box will be
white. Otherwise, a color string. Valid colors are listed in
\link{validColors}.}

\item{width}{The width of the box, using the Bootstrap grid system. This is
used for row-based layouts. The overall width of a region is 12, so the
default valueBox width of 4 occupies 1/3 of that width. For column-based
layouts, use \code{NULL} for the width; the width is set by the column that
contains the box.}

\item{height}{The height of a box, in pixels or other CSS unit. By default
the height scales automatically with the content.}

\item{collapsible}{If TRUE, display a button in the upper right that allows
the user to collapse the box.}

\item{collapsed}{If TRUE, start collapsed. This must be used with
\code{collapsible=TRUE}.}
}
\description{
Boxes can be used to hold content in the main body of a dashboard.
}
\examples{
## Only run this example in interactive R sessions
if (interactive()) {
library(shiny)

# A dashboard body with a row of infoBoxes and valueBoxes, and two rows of boxes
body <- dashboardBody(

  # infoBoxes
  fluidRow(
    infoBox(
      "Orders", uiOutput("orderNum2"), "Subtitle", icon = icon("credit-card")
    ),
    infoBox(
      "Approval Rating", "60\%", icon = icon("line-chart"), color = "green",
      fill = TRUE
    ),
    infoBox(
      "Progress", uiOutput("progress2"), icon = icon("users"), color = "purple"
    )
  ),

  # valueBoxes
  fluidRow(
    valueBox(
      uiOutput("orderNum"), "New Orders", icon = icon("credit-card"),
      href = "http://google.com"
    ),
    valueBox(
      tagList("60", tags$sup(style="font-size: 20px", "\%")),
       "Approval Rating", icon = icon("line-chart"), color = "green"
    ),
    valueBox(
      htmlOutput("progress"), "Progress", icon = icon("users"), color = "purple"
    )
  ),

  # Boxes
  fluidRow(
    box(status = "primary",
      sliderInput("orders", "Orders", min = 1, max = 2000, value = 650),
      selectInput("progress", "Progress",
        choices = c("0\%" = 0, "20\%" = 20, "40\%" = 40, "60\%" = 60, "80\%" = 80,
                    "100\%" = 100)
      )
    ),
    box(title = "Histogram box title",
      status = "warning", solidHeader = TRUE, collapsible = TRUE,
      plotOutput("plot", height = 250)
    )
  ),

  # Boxes with solid color, using `background`
  fluidRow(
    # Box with textOutput
    box(
      title = "Status summary",
      background = "green",
      width = 4,
      textOutput("status")
    ),

    # Box with HTML output, when finer control over appearance is needed
    box(
      title = "Status summary 2",
      width = 4,
      background = "red",
      uiOutput("status2")
    ),

    box(
      width = 4,
      background = "light-blue",
      p("This is content. The background color is set to light-blue")
    )
  )
)

server <- function(input, output) {
  output$orderNum <- renderText({
    prettyNum(input$orders, big.mark=",")
  })

  output$orderNum2 <- renderText({
    prettyNum(input$orders, big.mark=",")
  })

  output$progress <- renderUI({
    tagList(input$progress, tags$sup(style="font-size: 20px", "\%"))
  })

  output$progress2 <- renderUI({
    paste0(input$progress, "\%")
  })

  output$status <- renderText({
    paste0("There are ", input$orders,
      " orders, and so the current progress is ", input$progress, "\%.")
  })

  output$status2 <- renderUI({
    iconName <- switch(input$progress,
      "100" = "ok",
      "0" = "remove",
      "road"
    )
    p("Current status is: ", icon(iconName, lib = "glyphicon"))
  })


  output$plot <- renderPlot({
    hist(rnorm(input$orders))
  })
}

shinyApp(
  ui = dashboardPage(
    dashboardHeader(),
    dashboardSidebar(),
    body
  ),
  server = server
)
}
}
\seealso{
Other boxes: \code{\link{infoBox}}, \code{\link{tabBox}},
  \code{\link{valueBox}}
}
