% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/semTable.R
\name{semTable}
\alias{semTable}
\title{Creates Structural Equation Modeling Tables}
\usage{
semTable(object, file = NULL, paramSets = "all", paramSetLabels,
  columns = c(est = "Estimate", se = "SE", z = "z", p = "p"),
  columnLabels, fits = c("chisq", "cfi", "tli", "rmsea"),
  fitLabels = toupper(fits), varLabels = NULL, groups = NULL,
  type = "latex", table.float = FALSE, caption = NULL,
  label = NULL, longtable = FALSE, print.results = TRUE,
  centering = "siunitx", alpha = c(0.05, 0.01, 0.001))
}
\arguments{
\item{object}{A lavaan object (e.g., returned by cfa() or sem()),
or a named list of lavaan objects, e.g., \code{list("Model A"
= obj1, "Model B" = obj2)}. Results will be displayed side by
side.}

\item{file}{Base name for output file. This function will insert
suffix, either "tex", "html" and "csv".}

\item{paramSets}{Parameter sets to be included for each fitted
object.  Valid values of the vector are \code{"all"} or a any
of the following: \code{c("loadings", "slopes", "intercepts",
"residualvariances", "residualcovariances", "latentmeans",
"latentvariances", "latentcovariances", "thresholds",
"constructed", "fits")}. Default is "all", any of the
estimates present in the fitted model that are listed in the
previous sentence will be included in the output. For the sake
of simplicity, we now allow one vector here, which applies to
all models in the object list.}

\item{paramSetLabels}{Named vector, used to supply alternative
pretty printing labels for parameter sets. The default values
are \code{c("loadings"= "Factor Loadings", "slopes" =
"Regression Slopes", "intercepts" = "Intercepts", "means"=
"Means", "residualvariances" = "Residual Variances",
"residualcovariances" = "Residual Covariances", "variances" =
"Variances", "latentvariances" = "Latent Variances",
"latentcovariances" = "Latent Covariances", "latentmeans" =
"Latent Intercepts", "thresholds" = "Thresholds",
"constructed" = "Constructed", "fits" = "Fit Indices")}.  The
paramSetLabels argument must be a named vector that overrides
some or all of the default names.}

\item{columns}{A vector naming estimates to appear for each model.
The allowed columns are "est", "se", "z", "p", "rsquare",
"estse", "eststars", "estsestars". The first 5 have the usual
meanings, while "estse" (can also be written \code{"est(se)"})
displays as, for example "1.21(0.23)", and the last 2 are to
include "significance stars".  \code{"eststars"} shows as
"1.21***" and \code{"estsestars"} (or \code{"est(se)stars"})
displays as "1.21(0.23)**". See parameter \code{alpha}. One
may request different columns for each model by providing a
named list of vectors.  Use model names in the list,
\code{list("Model A" = c("est", "se"), "Model B" = c("estse",
"p"))}.}

\item{columnLabels}{A named vector of "pretty labels" for the
headings in the table. The default labels are \code{c("est" =
"Estimate", se = "Std. Err.", z = "z", p = "p", rsquare =
"R Square", estse = "Estimate(Std.Err."), eststars =
"Estimate", estsestars = "Estimate(Std.Err.)")}.}

\item{fits}{Summary indicators to be included. May be a list, one
for each model provided, otherwise the same fit indicators
will be presented for each model. Any of the fit indicators
provided by \code{lavaan::fitMeasures(object)} are allowed:
\code{c("npar", "fmin", "chisq", "df", "pvalue",
"baseline.chisq", "baseline.df", "baseline.pvalue", "cfi",
"tli", "nnfi", "rfi", "nfi", "pnfi", "ifi", "rni", "logl",
"unrestricted.logl", "aic", "bic", "ntotal", "bic2", "rmsea",
"rmsea.ci.lower", "rmsea.ci.upper", "rmsea.pvalue", "rmr",
"rmr_nomean", "srmr", "srmr_bentler", "srmr_bentler_nomean",
"srmr_bollen", "srmr_bollen_nomean", "srmr_mplus",
"srmr_mplus_nomean", "cn_05", "cn_01", "gfi", "agfi", "pgfi",
"mfi", "ecvi")}. The return for "chisq" will include markup
for degrees of freedom and p value. If user specifies
\code{NULL}, or if "fits" is excluded from \code{paramSets},
all fit indicators are omitted.}

\item{fitLabels}{Labels for some or all of the fit measures
requested by the fits parameter, e.g. \code{c(rmsea =
"Root Mean Square Error of Approximation", cli = "CLI")}. The
default labels are the upper-case fits names (except for
"chisq", where a Greek letter is supplied when possible).}

\item{varLabels}{Named vector of labels to replace variable names
in column 1 of SEM table.}

\item{groups}{All groups will be printed, unless a subset is
requested here. Estimates for all groups will be displayed
side by side. If ONLY SOME groups should be included, then
specify groups as either names of fit objects or as integers
for elements of the groups vector.}

\item{type}{Choose "latex", "html", or "csv"}

\item{table.float}{If TRUE, create a LaTeX floating table object
in which the tabular created here will reside. Default is
FALSE.}

\item{caption}{Caption for table (if table.float=TRUE) or
longtable output. Ignored otherwise.}

\item{label}{LaTeX label for this object (for
cross-references). Only used if table.float = TRUE or
longtable = TRUE.}

\item{longtable}{If TRUE, use longtable for LaTeX
documents. Default is FALSE. If true, \code{table.float}
argument is ignored.}

\item{print.results}{If TRUE, marked up result will be displayed
within the session. Otherwise, result is returned silently and
user can use \code{cat} to dislay it. Don't use \code{print}
because it inserts unwanted decorations.}

\item{centering}{Default "siunitx". For method used in previous
editions, replace with "none".}

\item{alpha}{Thresholds for p-values that determine number of
stars.  Defaults as \code{c(0.05, 0.01, 0.001)} for
\code{c("*", "**", "***")}.}
}
\value{
Markup for SEM table. Includes an attribute
    "markedResults", which can be converted to other markup
    formats by the function markupConvert.
}
\description{
Creates LaTeX markup for structural equation modeling output
tables in the style of the American Psychological
Association(APA). Input objects should be created by the
"\code{lavaan}" package.
}
\details{
The argument paramSets determines the inclusion of estimate sections.
One can specify "all", which means that all types of parameters that
we can find in the fitted model are presented.  Otherwise, a subset
of parameter sets can be chosen by the user.
\itemize{
\item "loadings" are the factor loadings in the model.
\item "slopes" are the regression slopes in the model.
\item "intercepts" are the estimated constants in the measurement
     models.
\item "residualvariances" are the observed variable residual variances.
\item "residualcovariances" are the observed covariances among
residuals of observed variables.
\item "latentvariances" are the variances of unobserved variables.
\item "latentcovariances" are the covariances between unobserved variables.
\item "latentmeans" are means of unobserved variables
\item "thresholds" arise in latent
response variates (non-numeric indicator data).
\item "constructed" are parameters that are calculated from a formula
      in the model specification, such as an indirect path c=a*b.
\item "fits" the summary indicators of the mismatch between
the theoretical and observed covariance matrices, such as
RMSEA, CLI, TFI. While the fits are not technically parameters, they
are displayed in the same block style as parameters
}

The columns parameter is used to specify different columns,
while columnLabels will alter the displayed labels for them.
}
\examples{
## Most of the examples were moved to the semTable vignette
require(lavaan)

tempdir <- tempdir()
## The example from lavaan's docs
 HS.model <- ' visual  =~ x1 + x2 + x3
               textual =~ x4 + x5 + x6
               speed   =~ x7 + x8 + x9'
 fit1 <- cfa(HS.model, data = HolzingerSwineford1939,
             std.lv = TRUE, meanstructure = TRUE)
 ## Try a LaTeX file first
 fit1.t1 <- semTable(fit1, columns = c("estse", "p"),
                     fits = c("chisq", "rmsea"), file = file.path(tempdir, "fit1.t1"),
                     varLabels = c("x1" = "hello"), type = "latex", print.results = FALSE)
 ## If you have a working version of pdflatex in your system path, 
 if (interactive()) testtable("fit1.t1.tex", tempdir)
 
 model <- "factor =~ .7*y1 + .7*y2 + .7*y3 + .7*y4
                  y1 | -1*t1 + 1*t2
                  y2 | -.5*t1 + 1*t2
                  y3 | -.2*t1 + 1*t2
                  y4 | -1*t1 + 1*t2"
 dat <- simulateData(model, sample.nobs = 300)
 
 testmodel <- "ExampleFactor =~ y1 + y2 + y3 + y4"
 
 fit4 <- cfa(testmodel, data = dat, ordered = colnames(dat),
             std.lv = FALSE)
 
 fit4.t1 <- semTable(fit4, paramSets = c("loadings", "thresholds",
     "residualvariances"), fits = c("tli", "chisq"),
     fitLabels = c(tli = "TLI", chisq = "chisq"), type = "html",
     file=file.path(tempdir, "fit4.t1") )
 if(interactive()) browseURL(attr(fit4.t1, "file"))
 fit4.t2 <- semTable(fit4, fits = c("rmsea", "cfi", "chisq"),
               fitLabels = c(rmsea = "Root M.SQ.E.A", cfi = "CompFitIdx", chisq = "chisq"),
               type = "latex", file=file.path(tempdir, "fit4.t2"))
 if (interactive()) testtable("fit4.t2.tex", tempdir)
}
\author{
Ben Kite <bakite@ku.edu> Paul Johnson <pauljohn@ku.edu>
}
