% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/convenienceFunctions.R
\name{semPower.powerRICLPM}
\alias{semPower.powerRICLPM}
\title{semPower.powerRICLPM}
\usage{
semPower.powerRICLPM(
  type,
  comparison = "restricted",
  nWaves = NULL,
  autoregEffects = NULL,
  crossedEffects = NULL,
  rXY = NULL,
  rBXBY = NULL,
  waveEqual = NULL,
  nullEffect = NULL,
  nullWhichGroups = NULL,
  nullWhich = NULL,
  standardized = TRUE,
  metricInvariance = TRUE,
  autocorResiduals = TRUE,
  ...
)
}
\arguments{
\item{type}{type of power analysis, one of \code{'a-priori'}, \code{'post-hoc'}, \code{'compromise'}.}

\item{comparison}{comparison model, one of \code{'saturated'} or \code{'restricted'} (the default). This determines the df for power analyses. \code{'saturated'} provides power to reject the model when compared to the saturated model, so the df equal the one of the hypothesized model. \code{'restricted'} provides power to reject the hypothesized model when compared to an otherwise identical model that just omits the restrictions defined in \code{nullEffect}, so the df equal the number of restrictions.}

\item{nWaves}{number of waves, must be >= 3.}

\item{autoregEffects}{vector of the autoregressive effects of X and Y (constant across waves), or a list of vectors of autoregressive effects for X and Y from wave to wave, e.g. \code{list(c(.7, .6), c(.5, .5))} for an autoregressive effect of .7 for X1->X2 and .6 for X2->X3 and autoregressive effects of .5 for Y1->Y2 and Y2->Y3. Must be a list of lists for multiple groups models. If the list structure is omitted, no group differences are assumed.}

\item{crossedEffects}{vector of crossed effects of X on Y (X -> Y) and vice versa (both constant across waves), or a list of vectors of crossed effects giving the crossed effect of X on Y (and vice versa) for each wave, e.g. \code{list(c(.2, .3), c(.1, .1))} for X1->Y2 = .2, X2->Y3 = .3, Y1->Y2 = .1, and Y2->Y3 = .1. Must be a list of lists for multiple groups models. If the list structure is omitted, no group differences are assumed.}

\item{rXY}{vector of (residual-)correlations between X and Y for each wave. If \code{NULL}, all (residual-)correlations are zero.  Can be a list for multiple groups models, otherwise no group differences are assumed.}

\item{rBXBY}{correlation between random intercept factors. If \code{NULL}, the correlation is zero. Must be a list of lists for multiple groups models. If the list structure is omitted, no group differences are assumed.}

\item{waveEqual}{parameters that are assumed to be equal across waves in both the H0 and the H1 model. Valid are \code{'autoregX'} and \code{'autoregY'} for autoregressive effects, \code{'crossedX'} and \code{'crossedY'} for crossed effects, \code{'corXY'} for residual correlations, or \code{NULL} for none (so that all parameters are freely estimated, subject to the constraints defined in \code{nullEffect}).}

\item{nullEffect}{defines the hypothesis of interest. Valid are the same arguments as in \code{waveEqual} and additionally \code{'autoregX = 0'}, \code{'autoregY = 0'}, \code{'crossedX = 0'}, \code{'crossedY = 0'} to constrain the X or Y autoregressive effects or the crossed effects to zero, \code{'corBXBY = 0'} to constrain the correlation between the random intercepts to zero, and \code{'autoregX = autoregY'} and \code{'crossedX = crossedY'} to constrain them to be equal for X and Y, and \code{'autoregXA = autoregXB'}, \code{'autoregYA = autoregYB'}, \code{'crossedXA = crossedXB'}, \code{'crossedYA = crossedYB'}, and \code{corBXBYA = corBXBYB} to constrain them to be equal across groups.}

\item{nullWhichGroups}{for hypothesis involving cross-groups comparisons, vector indicating the groups for which equality constrains should be applied, e.g. \code{c(1, 3)} to constrain the relevant parameters of the first and the third group. If \code{NULL}, all groups are constrained to equality.}

\item{nullWhich}{used in conjunction with \code{nullEffect} to identify which parameter to constrain when there are > 2 waves and parameters are not constant across waves. For example, \code{nullEffect = 'autoregX = 0'} with \code{nullWhich = 2} would constrain the second autoregressive effect for X to zero.}

\item{standardized}{whether the autoregressive and cross-lagged parameters should be treated as standardized (\code{TRUE}, the default), implying that unstandardized and standardized regression relations have the same value. If \code{FALSE}, all regression relations are unstandardized.}

\item{metricInvariance}{whether metric invariance over waves is assumed (\code{TRUE}, the default) or not (\code{FALSE}). This affects the df when the comparison model is the saturated model and generally affects power (also for comparisons to the restricted model, where the df are not affected by invariance constraints).}

\item{autocorResiduals}{whether the residuals of the indicators of latent variables are autocorrelated over waves (\code{TRUE}, the default) or not (\code{FALSE}). This affects the df when the comparison model is the saturated model and generally affects power (also for comparisons to the restricted model).}

\item{...}{mandatory further parameters related to the specific type of power analysis requested, see \code{\link[=semPower.aPriori]{semPower.aPriori()}}, \code{\link[=semPower.postHoc]{semPower.postHoc()}}, and \code{\link[=semPower.compromise]{semPower.compromise()}}, and parameters specifying the factor model. The order of factors is (X1, Y1, X2, Y2, ..., X_nWaves, Y_nWaves). See details.}
}
\value{
a list. Use the \code{summary} method to obtain formatted results. Beyond the results of the power analysis and a number of effect size measures, the list contains the following components:
\item{\code{Sigma}}{the population covariance matrix. A list for multiple group models.}
\item{\code{mu}}{the population mean vector or \code{NULL} when no meanstructure is involved. A list for multiple group models.}
\item{\code{SigmaHat}}{the H0 model implied covariance matrix. A list for multiple group models.}
\item{\code{muHat}}{the H0 model implied mean vector or \code{NULL} when no meanstructure is involved. A list for multiple group models.}
\item{\code{modelH0}}{\code{lavaan} H0 model string.}
\item{\code{modelH1}}{\code{lavaan} H1 model string or \code{NULL} when the comparison refers to the saturated model.}
\item{\code{simRes}}{detailed simulation results when a simulated power analysis (\code{simulatedPower = TRUE}) was performed.}
}
\description{
Convenience function for performing power analysis on effects in a random intercept cross-lagged panel model (RI-CLPM).
This requires the lavaan package.
}
\details{
This function performs a power analysis to reject various hypotheses arising in a random intercept crossed-lagged panel model (RI-CLPM).
In a standard RI-CLPM implemented here, two variables X and Y are repeatedly assessed at three or more different time points (\code{nWaves}),
yielding autoregressive effects (\code{X1 -> X2}, \code{X2 -> X3}, \code{Y1 -> Y2}, \code{Y2 -> Y3}), synchronous effects (\verb{X1 <-> Y1}, \verb{X2 <-> Y2}, \verb{X3 <-> Y3}), and cross-lagged effects (\code{X1 -> Y2}, \code{X2 -> Y3}, \code{Y1 -> X2}, \code{Y2 -> X3}).
RI-CLPMs are typically implemented assuming that the parameters are constant across waves (\code{waveEqual}), and usually omit lag-2 effects (e.g., \code{X1 -> Y3}).
RI-CLPMs based on latent factors usually assume at least metric invariance of the factors over waves (\code{metricInvariance}).

Relevant hypotheses in arising in a RI-CLPM are:
\itemize{
\item \code{autoregX = 0} and \code{autoregY = 0}: Tests the hypothesis that the autoregressive effect of X and Y, respectively, is zero.
\item \code{crossedX = 0} and \code{crossedY = 0}: Tests the hypothesis that the crossed effect of X on Y (\code{crossedX}) and Y on X (\code{crossedY}), respectively, is zero.
\item \code{autoregX = autoregY}: Tests the hypothesis that the autoregressive effect of X and Y are equal.
\item \code{crossedX = crossedY}: Tests the hypothesis that the crossed effect of X on Y (\code{crossedX}) and Y on X (\code{crossedY}) are equal.
\item \code{autoregX} and \code{autoregY}: Tests the hypothesis that the autoregressive effect of X and Y, respectively, is equal across waves.
\item \code{crossedX} and \code{crossedY}: Tests the hypothesis that the crossed effect of X on Y (\code{crossedX}) and Y on X (\code{crossedY}), respectively, is equal across waves.
\item \code{corXY}: Tests the hypothesis that the (residual-)correlations between X and Y are equal across waves.
\item \code{corBXBY = 0}: Tests the hypothesis that the correlation between the random intercept factors of X and Y is zero.
\item \code{autoregXA = autoregXB} and \code{autoregYA = autoregYB}: Tests the hypothesis that the autoregressive effect of either X or Y are equal across groups.
\item \code{crossedXA = crossedXB} and \code{crossedYA = crossedYB}: Tests the hypothesis that the crossed effect of X on Y (\code{crossedX}) or of Y on X (\code{crossedY}), respectively, is equal across groups.
\item \code{corBXBYA = corBXBYB}: Tests the hypothesis that the correlation between the random intercept factors is equal across groups.
}

For hypotheses regarding the traditional CLPM, see \code{\link[=semPower.powerCLPM]{semPower.powerCLPM()}}.

Beyond the arguments explicitly contained in the function call, additional arguments
are required specifying the factor model and the requested type of power analysis.

Additional arguments related to the \strong{definition of the factor model}:
\itemize{
\item \code{Lambda}: The factor loading matrix (with the number of columns equaling 2 times the number of waves). Columns should be in order X1, Y1, X2, Y2, ..., X_nWaves, Y_nWaves.
\item \code{loadings}: Can be used instead of \code{Lambda}: Defines the primary loadings for each factor in a list structure ordered by wave, e.g., list(c(.2, .2, .2), c(.4, .4, .4, .4), c(.2, .2, .2), c(.4, .4, .4, .4), c(.2, .2, .2), c(.4, .4, .4, .4)) defines loadings of .2 for the three indicators of X at waves 1-3 and loadings of .4 for the four indicators of Y at waves 1-3. Must not contain secondary loadings.
\item \code{nIndicator}: Can be used instead of \code{Lambda}: Used in conjunction with \code{loadM}. Defines the number of indicators for each factor ordered by wave, e.g. c(3, 4, 3, 4, 3, 4) defines three indicators for X at waves 1-3 and four indicators for Y at waves 1-3.
\item \code{loadM}: Can be used instead of \code{Lambda}: Used in conjunction with \code{nIndicator}. Defines the loading either for all indicators (if a single number is provided) or separately for each factor at each wave (if a vector is provided), e. g. \code{loadM = c(.5, .6, .5, .6, .5, .6)} defines mean loadings of .5 for X at waves 1-3 and mean loadings of .6 for Y at waves 1-3.
}

So either \code{Lambda}, or \code{loadings}, or \code{nIndicator} and \code{loadM} need to be defined.
If the model contains observed variables only, use \code{Lambda = diag(x)} where \code{x} is the number of variables.

Note that the order of the factors is (X1, Y1, X2, Y2, ..., X_nWaves, Y_nWaves), i. e., the first factor is treated as the first measurement of X, the second as the first measurement of Y, the third as the second measurement of X, etc..

Additional arguments related to the requested type of \strong{power analysis}:
\itemize{
\item \code{alpha}: The alpha error probability. Required for \code{type = 'a-priori'} and \code{type = 'post-hoc'}.
\item Either \code{beta} or \code{power}: The beta error probability and the statistical power (1 - beta), respectively. Only for \code{type = 'a-priori'}.
\item \code{N}: The sample size. Always required for \code{type = 'post-hoc'} and \code{type = 'compromise'}. For \code{type = 'a-priori'} and multiple group analysis, \code{N} is a list of group weights.
\item \code{abratio}: The ratio of alpha to beta. Only for \code{type = 'compromise'}.
}

If a \strong{simulated power analysis} (\code{simulatedPower = TRUE}) is requested, optional arguments can be provided as a list to \code{simOptions}:
\itemize{
\item \code{nReplications}: The targeted number of simulation runs. Defaults to 250, but larger numbers greatly improve accuracy at the expense of increased computation time.
\item \code{minConvergenceRate}:  The minimum convergence rate required, defaults to .5. The maximum actual simulation runs are increased by a factor of 1/minConvergenceRate.
\item \code{type}: specifies whether the data should be generated from a population assuming multivariate normality (\code{'normal'}; the default), or based on an approach generating non-normal data (\code{'IG'}, \code{'mnonr'}, \code{'RC'}, or \code{'VM'}).
The approaches generating non-normal data require additional arguments detailed below.
\item \code{missingVars}: vector specifying the variables containing missing data (defaults to NULL).
\item \code{missingVarProp}: can be used instead of \code{missingVars}: The proportion of variables containing missing data (defaults to zero).
\item \code{missingProp}: The proportion of missingness for variables containing missing data (defaults to zero), either a single value or a vector giving the probabilities for each variable.
\item \code{missingMechanism}: The missing data mechanism, one of \code{MCAR} (the default), \code{MAR}, or \code{NMAR}.
\item \code{nCores}: The number of cores to use for parallel processing. Defaults to 1 (= no parallel processing). This requires the \code{doSNOW} package.
}

\code{type = 'IG'} implements the independent generator approach (IG, Foldnes & Olsson, 2016) approach
specifying third and fourth moments of the marginals, and thus requires that skewness (\code{skewness}) and excess kurtosis (\code{kurtosis}) for each variable are provided as vectors. This requires the \code{covsim} package.

\code{type = 'mnonr'} implements the approach suggested by Qu, Liu, & Zhang (2020) and requires provision of  Mardia's multivariate skewness (\code{skewness})  and kurtosis  (\code{kurtosis}), where
skewness must be non-negative and kurtosis must be at least 1.641 skewness + p (p + 0.774), where p is the number of variables. This requires the \code{mnonr} package.

\code{type = 'RK'} implements the approach suggested by Ruscio & Kaczetow (2008) and requires provision of the population distributions
of each variable (\code{distributions}). \code{distributions} must be a list (if all variables shall be based on the same population distribution) or a list of lists.
Each component must specify the population distribution (e.g. \code{rchisq}) and additional arguments (\code{list(df = 2)}).

\code{type = 'VM'} implements the third-order polynomial method (Vale & Maurelli, 1983)
specifying third and fourth moments of the marginals, and thus requires that skewness (\code{skewness}) and excess kurtosis (\code{kurtosis}) for each variable are provided as vectors.  This requires the \code{semTools} package.
}
\examples{
\dontrun{
# Determine required N in a 3-wave RI-CLPM
# to detect crossed effects of X (X1 -> Y2 and X2 -> Y3) of >= .2
# with a power of 95\% on alpha = 5\%, where
# X1, X2, and X3 are measured by 5 indicators loading by .5 each, and
# Y1, Y2, and Y3 are measured by 3 indicators loading by .4 each, and
# there is no synchronous correlation between X and Y (rXY = NULL),
# the correlation between the random intercept factors of X and Y (rBXBY) is .1,
# the autoregressive effects of X are .8 (equal across waves),
# the autoregressive effects of Y are .7 (equal across waves), and
# the crossed effects of Y (Y1 -> X2 and Y2 -> X3) are .1 (equal across waves).

powerRICLPM <- semPower.powerRICLPM(type = 'a-priori',
                                    nWaves = 3,
                                    autoregEffects = c(.8, .7),
                                    crossedEffects = c(.2, .1),
                                    waveEqual = c('autoregX', 'autoregY', 
                                                  'crossedX', 'crossedY'),
                                    rXY = NULL,
                                    rBXBY = .1,
                                    nullEffect = 'crossedX = 0',
                                    nIndicator = c(5, 3, 5, 3, 5, 3),
                                    loadM = c(.5, .4, .5, .4, .5, .4),
                                    alpha = .05, beta = .05)

# show summary
summary(powerRICLPM)
# optionally use lavaan to verify the model was set-up as intended
lavaan::sem(powerRICLPM$modelH1, sample.cov = powerRICLPM$Sigma,
            sample.nobs = powerRICLPM$requiredN, sample.cov.rescale = FALSE)
lavaan::sem(powerRICLPM$modelH0, sample.cov = powerRICLPM$Sigma,
            sample.nobs = powerRICLPM$requiredN, sample.cov.rescale = FALSE)


# same as above, but determine power with N = 500 on alpha = .05
powerRICLPM <- semPower.powerRICLPM(type = 'post-hoc',
                                    nWaves = 3,
                                    autoregEffects = c(.8, .7),
                                    crossedEffects = c(.2, .1),
                                    waveEqual = c('autoregX', 'autoregY', 
                                                  'crossedX', 'crossedY'),
                                    rXY = NULL,
                                    rBXBY = .1,
                                    nullEffect = 'crossedX = 0',
                                    nIndicator = c(5, 3, 5, 3, 5, 3),
                                    loadM = c(.5, .4, .5, .4, .5, .4),
                                    alpha = .05, N = 500)


# same as above, but determine the critical chi-square with N = 500 so that alpha = beta
powerRICLPM <- semPower.powerRICLPM(type = 'compromise',
                                    nWaves = 3,
                                    autoregEffects = c(.8, .7),
                                    crossedEffects = c(.2, .1),
                                    waveEqual = c('autoregX', 'autoregY', 
                                                  'crossedX', 'crossedY'),
                                    rXY = NULL,
                                    rBXBY = .1,
                                    nullEffect = 'crossedX = 0',
                                    nIndicator = c(5, 3, 5, 3, 5, 3),
                                    loadM = c(.5, .4, .5, .4, .5, .4),
                                    abratio = 1, N = 500)


# same as above, but compare to the saturated model
# (rather than to the less restricted model)
powerRICLPM <- semPower.powerRICLPM(type = 'compromise',
                                    comparison = 'saturated',
                                    nWaves = 3,
                                    autoregEffects = c(.8, .7),
                                    crossedEffects = c(.2, .1),
                                    waveEqual = c('autoregX', 'autoregY', 
                                                  'crossedX', 'crossedY'),
                                    rXY = NULL,
                                    rBXBY = .1,
                                    nullEffect = 'crossedX = 0',
                                    nIndicator = c(5, 3, 5, 3, 5, 3),
                                    loadM = c(.5, .4, .5, .4, .5, .4),
                                    abratio = 1, N = 500)


# same as above, but assume only observed variables
powerRICLPM <- semPower.powerRICLPM(type = 'a-priori',
                                    nWaves = 3,
                                    autoregEffects = c(.8, .7),
                                    crossedEffects = c(.2, .1),
                                    waveEqual = c('autoregX', 'autoregY', 
                                                  'crossedX', 'crossedY'),
                                    rXY = NULL,
                                    rBXBY = .1,
                                    nullEffect = 'crossedX = 0',
                                    Lambda = diag(6),
                                    alpha = .05, beta = .05)


# same as above, but provide reduced loadings matrix to define that
# X1, X2, and X3 are measured by 5 indicators each loading by .5, .4, .5, .4, .3
# Y1, Y2, and Y3 are measured by 3 indicators each loading by .4, .3, .2
powerRICLPM <- semPower.powerRICLPM(type = 'a-priori',
                                    nWaves = 3,
                                    autoregEffects = c(.8, .7),
                                    crossedEffects = c(.2, .1),
                                    waveEqual = c('autoregX', 'autoregY', 
                                                  'crossedX', 'crossedY'),
                                    rXY = NULL,
                                    rBXBY = .1,
                                    nullEffect = 'crossedX = 0',
                                    loadings = list(
                                      c(.5, .4, .5, .4, .3),    # X1
                                      c(.4, .3, .2),            # Y1
                                      c(.5, .4, .5, .4, .3),    # X2
                                      c(.4, .3, .2),            # Y2
                                      c(.5, .4, .5, .4, .3),    # X3
                                      c(.4, .3, .2)             # Y3
                                    ),
                                    alpha = .05, beta = .05)


# same as above, but do not assume metric invariance across waves
powerRICLPM <- semPower.powerRICLPM(type = 'a-priori',
                                    nWaves = 3,
                                    autoregEffects = c(.8, .7),
                                    crossedEffects = c(.2, .1),
                                    waveEqual = c('autoregX', 'autoregY', 
                                                  'crossedX', 'crossedY'),
                                    rXY = NULL,
                                    rBXBY = .1,
                                    nullEffect = 'crossedX = 0',
                                    nIndicator = c(5, 3, 5, 3, 5, 3),
                                    loadM = c(.5, .4, .5, .4, .5, .4),
                                    metricInvariance = FALSE,
                                    alpha = .05, beta = .05)


# same as above, but determine N to detect that the crossed effect of Y 
# (Y1 -> X2 and Y2 -> X3) is >= .1.
powerRICLPM <- semPower.powerRICLPM(type = 'a-priori',
                                    nWaves = 3,
                                    autoregEffects = c(.8, .7),
                                    crossedEffects = c(.2, .1),
                                    waveEqual = c('autoregX', 'autoregY', 
                                                  'crossedX', 'crossedY'),
                                    rXY = NULL,
                                    rBXBY = .1,
                                    nullEffect = 'crossedY = 0',
                                    nIndicator = c(5, 3, 5, 3, 5, 3),
                                    loadM = c(.5, .4, .5, .4, .5, .4),
                                    alpha = .05, beta = .05)


# same as above, but determine N to detect that the autoregressive effect 
# of X (X1 -> X2 and X2 -> X3) is >= .8.
powerRICLPM <- semPower.powerRICLPM(type = 'a-priori',
                                    nWaves = 3,
                                    autoregEffects = c(.8, .7),
                                    crossedEffects = c(.2, .1),
                                    waveEqual = c('autoregX', 'autoregY', 
                                                  'crossedX', 'crossedY'),
                                    rXY = NULL,
                                    rBXBY = .1,
                                    nullEffect = 'autoregX = 0',
                                    nIndicator = c(5, 3, 5, 3, 5, 3),
                                    loadM = c(.5, .4, .5, .4, .5, .4),
                                    alpha = .05, beta = .05)


# same as above, but determine N to detect that the autoregressive effect 
# of Y (Y1 -> Y2) is >= .7.
powerRICLPM <- semPower.powerRICLPM(type = 'a-priori',
                                    nWaves = 3,
                                    autoregEffects = c(.8, .7),
                                    crossedEffects = c(.2, .1),
                                    waveEqual = c('autoregX', 'autoregY', 
                                                  'crossedX', 'crossedY'),
                                    rXY = NULL,
                                    rBXBY = .1,
                                    nullEffect = 'autoregY = 0',
                                    nIndicator = c(5, 3, 5, 3, 5, 3),
                                    loadM = c(.5, .4, .5, .4, .5, .4),
                                    alpha = .05, beta = .05)


# same as above, but determine N to detect that
# the crossed effect of X (X1 -> Y2) of .2 differs from
# the crossed effect of Y (Y1 -> X2) of .1
powerRICLPM <- semPower.powerRICLPM(type = 'a-priori',
                                    nWaves = 3,
                                    autoregEffects = c(.8, .7),
                                    crossedEffects = c(.2, .1),
                                    waveEqual = c('autoregX', 'autoregY', 
                                                  'crossedX', 'crossedY'),
                                    rXY = NULL,
                                    rBXBY = .1,
                                    nullEffect = 'crossedX = crossedY',
                                    nIndicator = c(5, 3, 5, 3, 5, 3),
                                    loadM = c(.5, .4, .5, .4, .5, .4),
                                    alpha = .05, beta = .05)


# same as above, but determine N to detect that
# the autoregressive effect of X (X1 -> X2) of .8 differs from
# the autoregressive effect of Y (Y1 -> Y2) of .7
powerRICLPM <- semPower.powerRICLPM(type = 'a-priori',
                                    nWaves = 3,
                                    autoregEffects = c(.8, .7),
                                    crossedEffects = c(.2, .1),
                                    waveEqual = c('autoregX', 'autoregY', 
                                                  'crossedX', 'crossedY'),
                                    rXY = NULL,
                                    rBXBY = .1,
                                    nullEffect = 'autoregX = autoregY',
                                    nIndicator = c(5, 3, 5, 3, 5, 3),
                                    loadM = c(.5, .4, .5, .4, .5, .4),
                                    alpha = .05, beta = .05)


# same as above, but determine N to detect that the correlation between the 
# random intercept factors is >= .1
powerRICLPM <- semPower.powerRICLPM(type = 'a-priori',
                                    nWaves = 3,
                                    autoregEffects = c(.8, .7),
                                    crossedEffects = c(.2, .1),
                                    waveEqual = c('autoregX', 'autoregY', 
                                                  'crossedX', 'crossedY'),
                                    rXY = NULL,
                                    rBXBY = .1,
                                    nullEffect = 'corBXBY = 0',
                                    nIndicator = c(5, 3, 5, 3, 5, 3),
                                    loadM = c(.5, .4, .5, .4, .5, .4),
                                    alpha = .05, beta = .05)


# same as above, but assume that the synchronous (residual-)correlations between
#  X and Y are equal across waves, 
# namely a synchronous correlation of .05 at the first wave and residual correlations 
# of .05 at the second and third wave,
# and determine N to detect a crossed effect of X (X1 -> Y2 and X2 -> Y3) of >= .2
powerRICLPM <- semPower.powerRICLPM(type = 'a-priori',
                                    nWaves = 3,
                                    autoregEffects = c(.8, .7),
                                    crossedEffects = c(.2, .1),
                                    waveEqual = c('autoregX', 'autoregY', 
                                                  'crossedX', 'crossedY', 
                                                  'corXY'),
                                    rXY = c(.05, .05, .05),
                                    rBXBY = .1,
                                    nullEffect = 'crossedX = 0',
                                    nIndicator = c(5, 3, 5, 3, 5, 3),
                                    loadM = c(.5, .4, .5, .4, .5, .4),
                                    alpha = .05, beta = .05)


# same as above, but assume that the synchronous correlation between X and Y
# is .3 at the first wave, and the respective residual correlations are .2 at 
# the second wave and .3 at the third wave,
# and determine N to detect that the synchronous residual correlation at wave 2 is => .2.
powerRICLPM <- semPower.powerRICLPM(type = 'a-priori',
                                    nWaves = 3,
                                    autoregEffects = c(.8, .7),
                                    crossedEffects = c(.2, .1),
                                    waveEqual = c('autoregX', 'autoregY', 
                                                  'crossedX', 'crossedY'),
                                    rXY = c(.3, .2, .3),
                                    rBXBY = .1,
                                    nullEffect = 'corXY = 0',
                                    nullWhich = 2,
                                    nIndicator = c(5, 3, 5, 3, 5, 3),
                                    loadM = c(.5, .4, .5, .4, .5, .4),
                                    alpha = .05, beta = .05)


# Determine required N in a 3-wave RI-CLPM to detect that
# the crossed effect of X at wave 1 (X1 -> Y2) of .20 is equal to the
# the crossed effect of X at wave 2 (X2 -> Y3) of .05
# with a power of 95\% on alpha = 5\%, where
# the autoregressive effects of X and Y are equal over waves,
# X1, X2, and X3 are measured by 5 indicators loading by .5 each, and
# Y1, Y2, and Y3 are measured by 3 indicators loading by .4 each, and
# the synchronous correlation between X and Y are .2, .3, and .4 at the first, 
# second, and third wave, 
# the correlation between the random intercept factors of X and Y is .1, and
# the autoregressive effect of X is .8 across all three waves,
# the autoregressive effect of Y is .7 across all three waves, and
# the crossed effects of Y (Y1 -> X2, and Y2 -> Y3) are both .1 
# (but freely estimated for each wave).
powerRICLPM <- semPower.powerRICLPM(type = 'a-priori',
                                    nWaves = 3,
                                    autoregEffects = c(.8, .7),
                                    crossedEffects = list(
                                      # X   Y
                                      c(.20, .10),  # wave 1 -> wave 2
                                      c(.05, .10)), # wave 2 -> wave 3
                                    waveEqual = c('autoregX', 'autoregY'),
                                    rXY = c(.2, .3, .4),
                                    rBXBY = .1,
                                    nullEffect = 'crossedX',
                                    nIndicator = c(5, 3, 5, 3, 5, 3),
                                    loadM = c(.5, .4, .5, .4, .5, .4),
                                    alpha = .05, beta = .05)


# same as above, but determine N to detect that
# the crossed effect of X at wave 2 is >= .05.
powerRICLPM <- semPower.powerRICLPM(type = 'a-priori',
                                    nWaves = 3,
                                    autoregEffects = c(.8, .7),
                                    crossedEffects = list(
                                      # X   Y
                                      c(.20, .10),  # wave 1 -> wave 2
                                      c(.05, .10)), # wave 2 -> wave 3
                                    waveEqual = c('autoregX', 'autoregY'),
                                    rXY = c(.2, .3, .4),
                                    rBXBY = .1,
                                    nullEffect = 'crossedX = 0',
                                    nullWhich = 2,
                                    nIndicator = c(5, 3, 5, 3, 5, 3),
                                    loadM = c(.5, .4, .5, .4, .5, .4),
                                    alpha = .05, beta = .05)


# same as above, but determine N to detect that
# the residual correlation between X and Y at wave 2 (of .3) differs from
# the residual correlation between X and Y at wave 3 (of .4).
powerRICLPM <- semPower.powerRICLPM(type = 'a-priori',
                                    nWaves = 3,
                                    autoregEffects = c(.8, .7),
                                    crossedEffects = list(
                                      # X   Y
                                      c(.20, .10),  # wave 1 -> wave 2
                                      c(.05, .10)), # wave 2 -> wave 3
                                    waveEqual = c('autoregX', 'autoregY'),
                                    rXY = c(.2, .3, .4),
                                    rBXBY = .1,
                                    nullEffect = 'corXY',
                                    nIndicator = c(5, 3, 5, 3, 5, 3),
                                    loadM = c(.5, .4, .5, .4, .5, .4),
                                    alpha = .05, beta = .05)


# multigroup example
# Determine the achieved power N in a 3-wave RI-CLPM to detect that
# the crossed effect of X at wave 1 (X1 -> Y2) in group 1 of .25 differs
# from the crossed effect of X at wave 1 (X1 -> Y2) in group 2 of .15,
# where both groups comprise 500 observations and alpha = 5\%, and
# the measurement model is equal for both groups, and
# the crossed effects of X (X1 -> Y2, and X2 -> Y3) are .25 and .10 in the first group, 
# the crossed effects of X (X1 -> Y2, and X2 -> Y3) are .15 and .05 in the second group, 
# the crossed effects of Y (Y1 -> X2, and Y2 -> X3) are .05 and .15 in the first group, 
# the crossed effects of Y (Y1 -> X2, and Y2 -> X3) are .01 and .10 in the second group, and
# the autoregressive effects of X (of .5) and Y (of .4) are equal over waves and over groups 
# (but freely estimated in each group).
powerRICLPM <- semPower.powerRICLPM(type = 'post-hoc', alpha = .05, N = list(500, 500),
                                    nWaves = 3,
                                    autoregEffects = c(.5, .4), # group and wave constant 
                                    crossedEffects = list(
                                      # group 1
                                      list(
                                        c(.25, .10),   # X
                                        c(.05, .15)    # Y 
                                      ),
                                      # group 2
                                      list(
                                        c(.15, .05),   # X
                                        c(.01, .10)    # Y 
                                      )
                                    ),
                                    rXY = NULL,        # identity
                                    rBXBY = NULL,      # identity 
                                    nullEffect = 'crossedXA = crossedXB',
                                    nullWhich = 1,
                                    nIndicator = rep(3, 6), 
                                    loadM = c(.5, .6, .5, .6, .5, .6),
                                    metricInvariance = TRUE,
                                    waveEqual = c('autoregX', 'autoregY')
                                    )


# Request a simulated post-hoc power analysis with 500 replications
# to detect crossed effects of X (X1 -> Y2 and X2 -> Y3) of >= .2
# with a power of 95\% on alpha = 5\% in a RI-CLPM with 3 waves, 
# where there are only observed variables and 
# there is no synchronous correlation between X and Y (rXY = NULL),
# and no correlation between the random intercept factors of X and Y (rBXBY = NULL),
# the autoregressive effects of X are .8 (equal across waves),
# the autoregressive effects of Y are .7 (equal across waves), and
# the crossed effects of Y (Y1 -> X2 and Y2 -> X3) are .1 (equal across waves).
set.seed(300121)
powerRICLPM <- semPower.powerRICLPM(type = 'post-hoc',
                                    nWaves = 3,
                                    autoregEffects = c(.8, .7),
                                    crossedEffects = c(.2, .1),
                                    waveEqual = c('autoregX', 'autoregY', 
                                                  'crossedX', 'crossedY'),
                                    rXY = NULL,
                                    rBXBY = NULL,
                                    nullEffect = 'crossedX = 0',
                                    Lambda = diag(6),
                                    alpha = .05, N = 500,
                                    simulatedPower = TRUE, 
                                    simOptions = list(nReplications = 500))
}
}
\seealso{
\code{\link[=semPower.genSigma]{semPower.genSigma()}} \code{\link[=semPower.aPriori]{semPower.aPriori()}} \code{\link[=semPower.postHoc]{semPower.postHoc()}} \code{\link[=semPower.compromise]{semPower.compromise()}}
}
