% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ACC.R
\name{ACC}
\alias{ACC}
\title{Compute the spatial anomaly correlation coefficient between the forecast and corresponding observation}
\usage{
ACC(
  exp,
  obs,
  dat_dim = NULL,
  lat_dim = "lat",
  lon_dim = "lon",
  avg_dim = "sdate",
  memb_dim = "member",
  lat = NULL,
  lon = NULL,
  lonlatbox = NULL,
  alpha = 0.05,
  pval = TRUE,
  sign = FALSE,
  conf = TRUE,
  conftype = "parametric",
  ncores = NULL
)
}
\arguments{
\item{exp}{A numeric array of experimental anomalies with named dimensions.
The dimension must have at least 'lat_dim' and 'lon_dim'.}

\item{obs}{A numeric array of observational anomalies with named dimensions.
The dimension should be the same as 'exp' except the length of 'dat_dim' 
and 'memb_dim'.}

\item{dat_dim}{A character string indicating the name of dataset (nobs/nexp) 
dimension. The default value is NULL (no dataset).}

\item{lat_dim}{A character string indicating the name of the latitude
dimension of 'exp' and 'obs' along which ACC is computed. The default value
is 'lat'.}

\item{lon_dim}{A character string indicating the name of the longitude
dimension of 'exp' and 'obs' along which ACC is computed. The default value
is 'lon'.}

\item{avg_dim}{A character string indicating the name of the dimension to be
averaged, which is usually the time dimension. If no need to calculate mean
ACC, set as NULL. The default value is 'sdate'.}

\item{memb_dim}{A character string indicating the name of the member 
dimension. If the data are not ensemble ones, set as NULL. The default 
value is 'member'.}

\item{lat}{A vector of the latitudes of the exp/obs grids. It is used for
area weighting and when the domain of interested 'lonlatbox' is specified.}

\item{lon}{A vector of the longitudes of the exp/obs grids. Only required when
the domain of interested 'lonlatbox' is specified. The default value is 
NULL.}

\item{lonlatbox}{A numeric vector of 4 indicating the corners of the domain of
interested: c(lonmin, lonmax, latmin, latmax). The default value is NULL 
and the whole data will be used.}

\item{alpha}{A numeric indicating the significance level for the statistical
significance test. The default value is 0.05.}

\item{pval}{A logical value indicating whether to compute the p-value or not.
The default value is TRUE.}

\item{sign}{A logical value indicating whether to retrieve the statistical
significance of the test Ho: ACC = 0 based on 'alpha'. The default value is
FALSE.}

\item{conf}{A logical value indicating whether to retrieve the confidence 
intervals or not. The default value is TRUE.}

\item{conftype}{A charater string of "parametric" or "bootstrap". 
"parametric" provides a confidence interval for the ACC computed by a 
Fisher transformation and a significance level for the ACC from a one-sided
student-T distribution. "bootstrap" provides a confidence interval for the
ACC and MACC computed from bootstrapping on the members with 100 drawings 
with replacement. To guarantee the statistical robustness of the result, 
make sure that your experiment and observation always have the same number
of members. "bootstrap" requires 'memb_dim' has value. The default value is
'parametric'.}

\item{ncores}{An integer indicating the number of cores to use for parallel 
computation. The default value is NULL.}
}
\value{
A list containing the numeric arrays:\cr 
\item{acc}{
 The ACC with the dimensions c(nexp, nobs, the rest of the dimension except 
 lat_dim, lon_dim and memb_dim). nexp is the number of experiment (i.e., dat_dim in
 exp), and nobs is the number of observation (i.e., dat_dim in obs). If 
 dat_dim is NULL, nexp and nobs are omitted.
}
\item{macc}{
 The mean anomaly correlation coefficient with dimensions
 c(nexp, nobs, the rest of the dimension except lat_dim, lon_dim, memb_dim, and 
 avg_dim). Only present if 'avg_dim' is not NULL. If dat_dim is NULL, nexp 
 and nobs are omitted.
}
\item{conf.lower (if conftype = "parametric") or acc_conf.lower (if 
     conftype = "bootstrap")}{
 The lower confidence interval of ACC with the same dimensions as ACC. Only
 present if \code{conf = TRUE}.
}
\item{conf.upper (if conftype = "parametric") or acc_conf.upper (if 
     conftype = "bootstrap")}{
 The upper confidence interval of ACC with the same dimensions as ACC. Only 
 present if \code{conf = TRUE}.
}
\item{p.val}{
 The p-value with the same dimensions as ACC. Only present if 
 \code{pval = TRUE} and \code{conftype = "parametric"}.  
}
\item{$sign}{
 The statistical significance. Only present if \code{sign = TRUE}.
}
\item{macc_conf.lower}{
 The lower confidence interval of MACC with the same dimensions as MACC. 
 Only present if \code{conftype = "bootstrap"}.
}
\item{macc_conf.upper}{
 The upper confidence interval of MACC with the same dimensions as MACC. 
 Only present if \code{conftype = "bootstrap"}.
}
}
\description{
Calculate the spatial anomaly correlation coefficient (ACC) for the ensemble
mean of each model and the corresponding references over a spatial domain. It
can return a forecast time series if the data contain forest time dimension, 
and also the ACC mean over one dimension, e.g., start date dimension.
The domain of interest can be specified by providing the list of longitudes/
latitudes of the data together with the corners of the domain: lonlatbox = 
c(lonmin, lonmax, latmin, latmax). The data will be adjusted to have a spatial
mean of zero, then area weighting is applied. The formula is referenced from 
Wilks (2011; section 7.6.4; https://doi.org/10.1016/B978-0-12-385022-5.00008-7).
}
\examples{
 \dontshow{
startDates <- c('19851101', '19901101', '19951101', '20001101', '20051101')
sampleData <- s2dv:::.LoadSampleData('tos', c('experiment'),
                                               c('observation'), startDates,
                                               leadtimemin = 1,
                                               leadtimemax = 4,
                                               output = 'lonlat',
                                               latmin = 27, latmax = 48,
                                               lonmin = -12, lonmax = 40)
 }
sampleData$mod <- Season(sampleData$mod, monini = 11, moninf = 12, monsup = 2)
sampleData$obs <- Season(sampleData$obs, monini = 11, moninf = 12, monsup = 2) 
clim <- Clim(sampleData$mod, sampleData$obs)
ano_exp <- Ano(sampleData$mod, clim$clim_exp)
ano_obs <- Ano(sampleData$obs, clim$clim_obs)
acc <- ACC(ano_exp, ano_obs, lat = sampleData$lat, dat_dim = 'dataset')
acc_bootstrap <- ACC(ano_exp, ano_obs, conftype = 'bootstrap', 
                    lat = sampleData$lat, dat_dim = 'dataset')
# Combine acc results for PlotACC
res <- array(c(acc$conf.lower, acc$acc, acc$conf.upper, acc$p.val), 
            dim = c(dim(acc$acc), 4))
res_bootstrap <- array(c(acc$acc_conf.lower, acc$acc, acc$acc_conf.upper, acc$p.val),
                      dim = c(dim(acc$acc), 4))
 \donttest{
PlotACC(res, startDates)
PlotACC(res_bootstrap, startDates)
 }
}
\references{
Joliffe and Stephenson (2012). Forecast Verification: A 
 Practitioner's Guide in Atmospheric Science. Wiley-Blackwell.; 
 Wilks (2011; section 7.6.4; https://doi.org/10.1016/B978-0-12-385022-5.00008-7).
}
