% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rosenblatt.R
\name{rosenblatt}
\alias{rosenblatt}
\alias{inverse_rosenblatt}
\title{(Inverse) Rosenblatt transform}
\usage{
rosenblatt(x, model, cores = 1, randomize_discrete = TRUE)

inverse_rosenblatt(u, model, cores = 1)
}
\arguments{
\item{x}{matrix of evaluation points; must be in \eqn{(0, 1)^d} for copula
models.}

\item{model}{a model object; classes currently supported are
\code{bicop_dist()}, \code{vinecop_dist()}, and \code{vine_dist()}.}

\item{cores}{if \verb{>1}, computation is parallelized over \code{cores} batches (rows
of \code{u}).}

\item{randomize_discrete}{Whether to randomize the transform for discrete
variables; see Details.}

\item{u}{matrix of evaluation points; must be in \eqn{(0, 1)^d}.}
}
\description{
The Rosenblatt transform takes data generated from a model and turns it into
independent uniform variates, The inverse Rosenblatt transform computes
conditional quantiles and can be used simulate from a stochastic model,
see \emph{Details}.
}
\details{
The Rosenblatt transform (Rosenblatt, 1952) \eqn{U = T(V)} of a random vector
\eqn{V = (V_1,\ldots,V_d) ~ F} is defined as
\deqn{
  U_1= F(V_1), U_{2} = F(V_{2}|V_1), \ldots, U_d =F(V_d|V_1,\ldots,V_{d-1}),
}
where \eqn{F(v_k|v_1,\ldots,v_{k-1})} is the conditional distribution of
\eqn{V_k} given \eqn{V_1 \ldots, V_{k-1}, k = 2,\ldots,d}. The vector
\eqn{U  = (U_1, \dots, U_d)} then contains independent standard uniform
variables. The inverse operation
\deqn{
  V_1 = F^{-1}(U_1), V_{2} = F^{-1}(U_2|U_1), \ldots,
  V_d =F^{-1}(U_d|U_1,\ldots,U_{d-1}),
}
can be used to simulate from a distribution. For any copula \eqn{F}, if
\eqn{U} is a vector of independent random variables, \eqn{V = T^{-1}(U)} has
distribution \eqn{F}.

The formulas above assume a vine copula model with order \eqn{d, \dots, 1}.
More generally, \code{rosenblatt()} returns the variables
\deqn{
  U_{M[d + 1- j, j]}= F(V_{M[d - j + 1, j]} | V_{M[d - j, j]}, \dots, V_{M[1, j]}),
}
where \eqn{M} is the structure matrix. Similarly, \code{inverse_rosenblatt()}
returns
\deqn{
  V_{M[d + 1- j, j]}= F^{-1}(U_{M[d - j + 1, j]} | U_{M[d - j, j]}, \dots, U_{M[1, j]}).
}

If some variables have atoms, Brockwell (10.1016/j.spl.2007.02.008) proposed
a simple randomization scheme to ensure that output is still independent
uniform if the model is correct. The transformation reads
\deqn{ U_{M[d - j,
j]}= W_{d - j} F(V_{M[d - j, j]} | V_{M[d - j - 1, j - 1]}, \dots, V_{M[0,
0]}) + (1 - W_{d - j}) F^-(V_{M[d - j, j]} | V_{M[d - j - 1, j - 1]}, \dots,
V_{M[0, 0]}),
}
where \eqn{F^-}
is the left limit of the conditional cdf
and \eqn{W_1, \dots, W_d} are are independent standard uniform random
variables. This is used by default. If you are interested in the conditional
probabilities
\deqn{
 F(V_{M[d - j, j]} | V_{M[d - j - 1, j - 1]}, \dots, V_{M[0, 0]}),
}
set \code{randomize_discrete = FALSE}.
}
\examples{
# simulate data with some dependence
x <- replicate(3, rnorm(200))
x[, 2:3] <- x[, 2:3] + x[, 1]
pairs(x)

# estimate a vine distribution model
fit <- vine(x, copula_controls = list(family_set = "par"))

# transform into independent uniforms
u <- rosenblatt(x, fit)
pairs(u)

# inversion
pairs(inverse_rosenblatt(u, fit))

# works similarly for vinecop models
vc <- fit$copula
rosenblatt(pseudo_obs(x), vc)
}
