#' Save a PhyloPic image
#'
#' `save_phylopic` is a convenient function for saving a PhyloPic silhouette
#' fetched via \code{\link[rphylopic]{get_phylopic}}. Available formats for
#' saving are those available in the [grDevices] package (i.e. pdf, png, svg,
#' tiff, jpeg, and bmp).
#'
#' @param img \code{picture}. A [Picture][grImport2::Picture-class] object.
#'   For PhyloPic silhouettes, this is generated by
#'   \code{\link[rphylopic]{get_phylopic}} and
#'   \code{\link[rphylopic]{pick_phylopic}}.
#' @param path \code{character}. The output file path. The name of the file
#'   and file extension should be included (e.g. "./phylopic.png"). If this
#'   argument is `NULL` (default), the file is automatically saved in the
#'   current working directory as a png file.
#' @param bg \code{character}. The desired background color. By default this
#' argument is set to "transparent". Note: .jpeg files are always exported
#' with a background colour.
#' @param ... Additional arguments available for the called grDevices
#'   function. The called function is determined by the file extension used in
#'   `path` (e.g. \code{\link[grDevices]{png}}).
#'
#' @return An image is saved. Nothing is returned to the R interface.
#'
#' @details This function can be used to save a PhyloPic image obtained via
#'   the \code{\link[rphylopic]{get_phylopic}} or any object of class
#'   `Picture` from the grImport2 package. Any file format available in the
#'   [grDevices] package can be used (i.e. pdf, png, svg, tiff, jpeg, and
#'   bmp).
#'
#' @importFrom grDevices pdf png svg tiff jpeg bmp dev.off
#' @importFrom grImport2 grid.picture
#' @importFrom methods is
#' @export
#' @examples
#' \dontrun{
#' # Get image data
#' img <- get_phylopic("27356f15-3cf8-47e8-ab41-71c6260b2724")
#' # Save file
#' save_phylopic(img = img, path = "./phylopic.png",
#'               width = 500, height = 500)
#' }
save_phylopic <- function(img = NULL, path = NULL, bg = "transparent", ...) {
  # Error handling -----------------------------------------------------
  # Image checking
  if (is.null(img)) {
    stop("`img` is required.")
  }
  if (!is(img, "Picture")) {
    stop("`img` should be of class Picture.")
  }
  # If path is NULL use current working directory and svg
  if (is.null(path)) {
    warning("No path was provided. Using current working directory.")
    path <- paste0(getwd(), c("/phylopic.png"))
  }
  # Define available save formats
  available_formats <- c("pdf", "png", "svg", "tiff", "jpeg", "jpg", "bmp")
  # Get file type for function selection
  type <- tools::file_ext(path)
  # Check for output format
  if (!type %in% available_formats) {
    msg <- paste0(type, " format not supported by this function.",
                  " Use one of the following: ", toString(available_formats))
    stop(msg)
  }
  # Save image ---------------------------------------------------------
  # Generate function list of different format
  formats <- list(pdf = pdf,
                  png = png,
                  svg = svg,
                  tiff = tiff,
                  jpeg = jpeg,
                  jpg = jpeg,
                  bmp = bmp)
  # Filter for correct function
  fun <- formats[[type]]
  # Save file
  do.call(fun, list(filename = path, bg = bg, ...))
  grid.picture(img)
  invisible(dev.off())
}
