% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{async_query}
\alias{async_query}
\alias{async_status}
\alias{is_busy}
\alias{cancel}
\alias{finish_async}
\title{Asynchronous query processing}
\usage{
async_query(sql = "", pars = NULL)

async_status()

is_busy()

cancel()

finish_async()
}
\arguments{
\item{sql}{a query string}

\item{pars}{a vector of parameters}
}
\value{
\code{async_query}: true if query was successfully sent (an invalid query
will still return true)

\code{async_status}: a results status object, possibly indicating an
invalid query

\code{is_busy}: a boolean
}
\description{
Manage an asynchronous query
}
\details{
These functions expose the asynchronous query interface from
\code{libpq}. The function \code{async_query} issues a query. Its
call is identical to \code{\link{query}} except that it will return
immediately. When the issued command is ready, the function
\code{async_status} will return a query status object exactly
as \code{\link{query}}. Otherwise it will return \code{"BUSY"} to
indicate the server has not finished or \code{"DONE"} to indicate
there is nothing more to fetch.

If \code{async_status} does not return \code{"DONE"}, then
you should call \code{finish_async} to free pending results. Note
that a call to \code{finish_async} may block until the server is finished
processing the command. It calls \code{cancel} internally but there is
no guarantee the command will abort.

Any pending results will be lost if you call \code{\link{query}},
\code{\link{execute}} or \code{\link{fetch}} with a \code{sql} string prior
to \code{async_query} returning \code{DONE}. If you need to issue queries
while waiting on an async call, then use \code{\link{push_conn}} to save
the query state, \code{\link{connect}} to make a new connetion, and then
\code{\link{pop_conn}} followed by \code{async_status}.

\code{is_busy} is a slightly faster shortcut to check whether the
server has completed the query. You must still call \code{async_status} to
fetch the results.
}
\note{
In practice, you will be much better off using \code{\link{cursor}}
as that will usually return very quickly even for large queries, and has
the advantage of retrieving the results in chunks. You can call \code{cancel}
while a cursor is active. The cursor will return \code{PGRES_FATAL_ERROR} if
the \code{cancel} is effective. Alternately, issuing any query that sets the
result status will have the same effect as \code{finish_async}.
}
\examples{
\dontrun{
# create a database
createdb("rpgtesting")
connect("rpgtesting")
begin()

# write data frame contents
data(mtcars)
write_table(mtcars)

# async processing on smallish result
# this wont be interesting if your machine is very fast
async_query("SELECT a.* FROM mtcars a, mtcars b")
repeat
{
  status = async_status()
  if ( status != "BUSY" ) break
  cat("busy...\\n")
  Sys.sleep(1)
}
print(status)
head(fetch())
finish_async()
Sys.sleep(1)

# async processing on larger result
async_query("SELECT a.* FROM mtcars a, mtcars b, mtcars c")
count = 0
repeat
{
  status = async_status()
  if ( status == "BUSY" )
  {
    if ( count > 2 )
    {
      cat("calling cancel...\\n")
      cancel()
    }
  }
  else break
  cat("busy... \\n")
  Sys.sleep(1)
  count = count + 1
}
print(status)
finish_async()

# you can run multiple queries with async_query
rollback(); begin()
write_table(mtcars)
sql1 = "SELECT mpg FROM mtcars LIMIT 3"
sql2 = "SELECT cyl FROM mtcars LIMIT 4"
async_query(paste(sql1, sql2, sep = "; "))
while ( async_status() == "BUSY" ) NULL
fetch()
while ( is_busy() ) NULL
async_status()
fetch()
finish_async()

# issue an async query and come back later
async_query(sql1)
push_conn()
connect("rpgtesting")

# fails because of transaction isolation
fetch(sql2)
pop_conn()
async_status()

# results from sql1
fetch()

# this is automatic if you issue new queries
finish_async()

# cleanup
rollback()
disconnect()
dropdb("rpgtesting")} 

}
\author{
Timothy H. Keitt
}
