% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/NoiseKrigingClass.R
\name{NoiseKriging}
\alias{NoiseKriging}
\title{Create a \code{NoiseKriging} Object using \pkg{libKriging}}
\usage{
NoiseKriging(
  y,
  noise,
  X,
  kernel,
  regmodel = c("constant", "linear", "interactive"),
  normalize = FALSE,
  optim = c("BFGS", "none"),
  objective = c("LL", "LMP"),
  parameters = NULL
)
}
\arguments{
\item{y}{Numeric vector of response values.}

\item{noise}{Numeric vector of response variances.}

\item{X}{Numeric matrix of input design.}

\item{kernel}{Character defining the covariance model:
\code{"exp"}, \code{"gauss"}, \code{"matern3_2"}, \code{"matern5_2"}.}

\item{regmodel}{Universal NoiseKriging linear trend.}

\item{normalize}{Logical. If \code{TRUE} both the input matrix
\code{X} and the response \code{y} in normalized to take
values in the interval \eqn{[0, 1]}.}

\item{optim}{Character giving the Optimization method used to fit
hyper-parameters. Possible values are: \code{"BFGS"} and \code{"none"}, 
the later simply keeping
the values given in \code{parameters}. The method
\code{"BFGS"} uses the gradient of the objective.}

\item{objective}{Character giving the objective function to
optimize. Possible values are: \code{"LL"} for the
Log-Likelihood.}

\item{parameters}{Initial values for the hyper-parameters. When
provided this must be named list with elements \code{"sigma2"}
and \code{"theta"} containing the initial value(s) for the
variance and for the range parameters. If \code{theta} is a
matrix with more than one row, each row is used as a starting
point for optimization.}
}
\value{
An object with S3 class \code{"NoiseKriging"}. Should be used
    with its \code{predict}, \code{simulate}, \code{update}
    methods.
}
\description{
Create an object with S3 class \code{"NoiseKriging"} using
the \pkg{libKriging} library.
}
\details{
The hyper-parameters (variance and vector of correlation ranges)
are estimated thanks to the optimization of a criterion given by
\code{objective}, using the method given in \code{optim}.
}
\examples{
f <- function(x) 1 - 1 / 2 * (sin(12 * x) / (1 + x) + 2 * cos(7 * x) * x^5 + 0.7)
set.seed(123)
X <- as.matrix(runif(10))
y <- f(X) + X/10 * rnorm(nrow(X)) # add noise dep. on X
## fit and print
k <- NoiseKriging(y, noise=(X/10)^2, X, kernel = "matern3_2")
print(k)

x <- as.matrix(seq(from = 0, to = 1, length.out = 101))
p <- predict(k,x = x, stdev = TRUE, cov = FALSE)

plot(f)
points(X, y)
lines(x, p$mean, col = "blue")
polygon(c(x, rev(x)), c(p$mean - 2 * p$stdev, rev(p$mean + 2 * p$stdev)),
border = NA, col = rgb(0, 0, 1, 0.2))

s <- simulate(k, nsim = 10, seed = 123, x = x)

matlines(x, s, col = rgb(0, 0, 1, 0.2), type = "l", lty = 1)
}
\author{
Yann Richet \email{yann.richet@irsn.fr}
}
