#' Evaluate Rust code
#'
#' Compile and evaluate one or more Rust expressions. If the last
#' expression in the Rust code returns a value (i.e., does not end with
#' `;`), then this value is returned to R. The value returned does not need
#' to be of type `Robj`, as long as it can be cast into this type with
#' `.into()`. This conversion is done automatically, so you don't have to
#' worry about it in your code.
#' @param code Input rust code.
#' @param env The R environment in which the Rust code will be evaluated.
#' @param ... Other parameters handed off to [rust_function()].
#' @return The return value generated by the Rust code.
#' @examples
#' \dontrun{
#' # Rust code without return value, called only for its side effects
#' rust_eval(
#'   code = 'rprintln!("hello from Rust!");'
#' )
#'
#' # Rust code with return value
#' rust_eval(
#'   code = "
#'     let x = 5;
#'     let y = 7;
#'     let z = x * y;
#'     z // return to R; rust_eval() takes care of type conversion code
#'  "
#' )
#' }
#' @export
rust_eval <- function(code, env = parent.frame(), ...) {
  # make sure code is given as a single character string
  code <- glue_collapse(code, sep = "\n")

  # define to make R code check happy; is not used
  rextendr_rust_eval_fun <- function() {
    ui_throw("decoy function; should never be called.")
  }

  # wrap code into Rust function
  code_wrapped <- glue(r"(
fn rextendr_rust_eval_fun() -> Result<Robj> {{
  let x = {{
    {code}
  }};
  Ok(x.into())
}}
)")
  out <- rust_function(code = code_wrapped, ...)
  result <- do.call(rextendr_rust_eval_fun, list(), envir = env)
  dyn.unload(out[["path"]])

  # Attempt to figure out whether the Rust code returns a result or not,
  # and make the result invisible or not accordingly. This regex approach
  # is not perfect, but since it only affects the visibility of the result
  # that's Ok. Worst case scenario a result that should be invisible is
  # shown as visible.
  if (grepl(".*;\\s*$", code, perl = TRUE)) {
    invisible(result)
  } else {
    result
  }
}
