% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/peer_old.R
\name{peer_old}
\alias{peer_old}
\title{Functional Models with Structured Penalties}
\usage{
peer_old(Y, funcs, argvals = NULL, pentype = "Ridge", L.user = NULL,
  Q = NULL, phia = 10^3, se = FALSE, ...)
}
\arguments{
\item{Y}{vector of all outcomes}

\item{funcs}{matrix containing observed functional predictors as rows. Rows
with \code{NA} and \code{Inf} values will be deleted.}

\item{argvals}{matrix (or vector) of indices of evaluations of \eqn{X_i(t)}; i.e. a matrix with
\emph{i}th row \eqn{(t_{i1},.,t_{iJ})}}

\item{pentype}{type of penalty. It can be either decomposition based
penalty (\code{DECOMP}) or ridge (\code{RIDGE}) or second-order difference
penalty (\code{D2}) or any user defined penalty (\code{USER}). For
decomposition based penalty user need to specify Q matrix in Q argument
(see details). For user defined penalty user need to specify L matrix in L
argument (see details). For Ridge and second-order difference penalty,
specification for arguments L and Q will be ignored. Default is
\code{RIDGE}.}

\item{L.user}{penalty matrix. Need to be specified with
\code{pentype='USER'}. Number of columns need to be equal with number of
columns of matrix specified to \code{funcs}. Each row represents a
constraint on functional predictor. This argument will be ignored when
value of \code{pentype} is other than \code{USER}.}

\item{Q}{Q matrix to derive decomposition based penalty. Need to be
specified with \code{pentype='DECOMP'}. Number of columns need to be equal
with number of columns of matrix specified to \code{funcs}. Each row
represents a basis function where functional predictor is expected lie
according to prior belief. This argument will be ignored when value of
\code{pentype} is other than \code{DECOMP}.}

\item{phia}{Scalar value of a in decomposition based penalty. Need to be
specified with \code{pentype='DECOMP'}.}

\item{se}{logical; calculate standard error when \code{TRUE}.}

\item{...}{additional arguments passed to the \code{\link{lme}} function.}
}
\value{
a list containing: \item{fit}{result of the call to \code{lme}}
\item{fitted.vals }{predicted outcomes} \item{Gamma}{estimates with
standard error for regression function} \item{GammaHat}{estimates of
regression function} \item{se.Gamma}{standard error associated with
\code{GammaHat}} \item{AIC }{AIC value of fit (smaller is better)}
\item{BIC }{BIC value of fit (smaller is better)}
\item{logLik}{(restricted) log-likelihood at convergence}
\item{lambda}{estimates of smoothing parameter} \item{N}{number of
subjects} \item{K}{number of Sampling points in functional predictor}
\item{sigma}{estimated within-group error standard deviation.}
}
\description{
Implements functional model with structured penalties (Randolph et al.,
2012) with scalar outcome and single functional predictor through mixed
model equivalence.
}
\details{
If there are any missing or infinite values in \code{Y}, and \code{funcs},
the corresponding row (or observation) will be dropped. Neither \code{Q}
nor \code{L} may contain missing or infinite values.

\code{peer_old()} fits the following model:

\eqn{y_i=\int {W_i(s)\gamma(s) ds} + \epsilon_i}

where \eqn{\epsilon_i ~ N(0,\sigma^2)}.  For all the observations,
predictor function \eqn{W_i(s)} is evaluated at K sampling points. Here,
\eqn{\gamma (s)} denotes the regression function.

Values of \eqn{y_i} and \eqn{W_i(s)}are passed through argument Y and
funcs, respectively. Number of elements or rows in \code{Y} and
\code{funcs} need to be equal.

The estimate of regression functions \eqn{\gamma(s)} is obtained as
penalized estimated. Following 3 types of penalties can be used:

i.  Ridge: \eqn{I_K}

ii.  Second-order difference: [\eqn{d_{i,j}}] with \eqn{d_{i,i} = d_{i,i+2}
= 1, d_{i,i+1} = -2}, otherwise \eqn{d_{i,i} =0}

iii. Decomposition based penalty: \eqn{bP_Q+a(I-P_Q)} where \eqn{P_Q=
Q^T(QQ^T)^{-1}Q}

For Decomposition based penalty user need to specify
\code{pentype='DECOMP'} and associated Q matrix need to be passed through
\code{Q} argument.

Alternatively, user can pass directly penalty matrix through argument L.
For this user need to specify \code{pentype='USER'} and associated L matrix
need to be passed through \code{L} argument.

Default penalty is Ridge penalty and user needs to specify \code{RIDGE}.
For second-order difference penalty, user needs to specify \code{D2}.
}
\examples{

\dontrun{
#------------------------------------------------------------------------
# Example 1: Estimation with D2 penalty
#------------------------------------------------------------------------

## Load Data
data(DTI)

## Extract values for arguments for peer() from given data
cca = DTI$cca[which(DTI$case == 1),]
DTI = DTI[which(DTI$case == 1),]

##1.1 Fit the model
fit.cca.peer1 = peer(Y=DTI$pasat, funcs = cca, pentype='D2', se=TRUE)
plot(fit.cca.peer1)

#------------------------------------------------------------------------
# Example 2: Estimation with structured penalty (need structural
#            information about regression function or predictor function)
#------------------------------------------------------------------------

## Load Data
data(PEER.Sim)

## Extract values for arguments for peer() from given data
PEER.Sim1<- subset(PEER.Sim, t==0)
W<- PEER.Sim1$W
Y<- PEER.Sim1$Y

##Load Q matrix containing structural information
data(Q)

##2.1 Fit the model
Fit1<- peer(Y=Y, funcs=W, pentype='Decomp', Q=Q, se=TRUE)
plot(Fit1)
}

}
\references{
Kundu, M. G., Harezlak, J., and Randolph, T. W. (2012).
Longitudinal functional models with structured penalties (arXiv:1211.4763
[stat.AP]).

Randolph, T. W., Harezlak, J, and Feng, Z. (2012). Structured penalties for
functional linear models - partially empirical eigenvectors for regression.
\emph{Electronic Journal of Statistics}, 6, 323--353.
}
\seealso{
\code{lpeer}, \code{plot.peer}
}
\author{
Madan Gopal Kundu \email{mgkundu@iupui.edu}
}
