\name{redist.ipw}
\alias{redist.ipw}
\title{
Inverse probability reweighting for MCMC Redistricting
}
\description{
\code{redist.ipw} properly weights and resamples simulated redistricting plans so that the set of simulated plans resemble a random sample from the underlying distribution. \code{redist.ipw} is used to correct the sample when population parity, geographic compactness, or other constraints are implemented. 
}
\usage{
redist.ipw(algout,
           resampleconstraint = c("pop", "compact", "segregation", "similar"),
           targetbeta,
           targetpop = NULL,
           temper = 0)
}

\arguments{
\item{algout}{An object of class "redist".}
\item{resampleconstraint}{The constraint implemented in the simulations: one of "pop", "compact", "segregation", or "similar".}
\item{targetbeta}{The target value of the constraint.}
\item{targetpop}{The desired level of population parity. \code{targetpop} = 0.01 means that the desired distance from population parity is 1\%. The default is \code{NULL}.}
\item{temper}{A flag for whether simulated tempering was used to improve the mixing of the Markov Chain. The default is \code{1}.}
}

\details{
This function allows users to resample redistricting plans using inverse probability weighting techniques described in Rubin (1987). This techniques reweights and resamples redistricting plans so that the resulting sample is representative of a random sample from the uniform distribution.
}

\value{
\code{redist.ipw} returns an object of class "redist". The object
\code{redist} is a list that contains the folowing components (the
inclusion of some components is dependent on whether tempering
techniques are used):

\item{partitions}{Matrix of congressional district assignments generated by the algorithm. Each row corresponds to a geographic unit, and each column corresponds to a simulation.}
\item{distance_parity}{Vector containing the maximum distance from parity for a particular simulated redistricting plan.}
\item{mhdecisions}{A vector specifying whether a proposed redistricting plan was accepted (1) or rejected (0) in a given iteration.}
\item{mhprob}{A vector containing the Metropolis-Hastings acceptance probability for each iteration of the algorithm.}
\item{pparam}{A vector containing the draw of the \code{p} parameter for each simulation, which dictates the number of swaps attempted.}
\item{constraint_pop}{A vector containing the value of the population constraint for each accepted redistricting plan.}
\item{constraint_compact}{A vector containing the value of the compactness constraint for each accepted redistricting plan.}
\item{constraint_segregation}{A vector containing the value of the segregation constraint for each accepted redistricting plan.}
\item{constraint_similar}{A vector containing the value of the similarity constraint for each accepted redistricting plan.}
\item{beta_sequence}{A vector containing the value of beta for each iteration of the algorithm. Returned when tempering is being used.}
\item{mhdecisions_beta}{A vector specifying whether a proposed beta value was accepted (1) or rejected (0) in a given iteration of the algorithm. Returned when tempering is being used.}
\item{mhprob_beta}{A vector containing the Metropolis-Hastings acceptance probability for each iteration of the algorithm. Returned when tempering is being used.}

}
\references{
  Fifield, Benjamin, Michael Higgins, Kosuke Imai and Alexander Tarr. (2015) "A New Automated Redistricting Simulator Using Markov Chain Monte Carlo." Working 
  Paper. Available at \url{http://imai.princeton.edu/research/files/redist.pdf}.

Rubin, Donald. (1987) "Comment: A Noniterative Sampling/Importance Resampling Alternative to the
Data Augmentation Algorithm for Creating a Few Imputations when Fractions of Missing Information
are Modest: the SIR Algorithm." Journal of the American Statistical Association.
}

\seealso{
\code{\link{redist.mcmc}} to simulate redistricting plans using Markov Chain Monte Carlo techniques.

}

\examples{
\dontrun{
data(algdat.p20)

## Code to run the simulations in Figure 4 of Fifield, Higgins, 
## Imai and Tarr (2015)

## Get an initial partition
set.seed(1)
initcds <- algdat.p20$cdmat[,sample(1:ncol(algdat.p20$cdmat), 1)]

## Vector of beta weights
betaweights <- rep(NA, 10); for(i in 1:10){betaweights[i] <- 4^i}

## Run simulations - tempering population constraint
alg_253_20_st <- redist.mcmc(adjobj = algdat.p20$adjlist,
                             popvec = algdat.p20$precinct.data$pop,
                             initcds = initcds,
                             nsims = 10000,
                             betapop = -5.4,
                             betaweights = betaweights,
                             temperbetapop = 1)

## Resample using inverse probability weighting. 
## Target distance from parity is 20%
alg_253_20_st <- redist.ipw(alg_253_20_st,
                            resampleconstraint = "pop",
                            targetbeta = -5.4,
                            targetpop = .2,
                            temper = 1)

}
}

\keyword{inverseprobabilityweighting}
