#' @name deleteFiles
#' @title Delete a File attached to a Record
#' 
#' @description This function allows you to remove a document that has been 
#' attached to an individual record
#' 
#' @param rcon A REDCap connection object as generated by \code{redcapConnection}
#' @param record The record ID in which the desired file is stored. Must be length 1.
#' @param field The field name in which the file is stored. Must be length 1.
#' @param event The event name for the file.  Must be length 1.  
#'   This applies only to longitudinal projects.  If the event is not
#'   supplied for a longitudinal project, the API will return an error message.
#' @param repeat_instance The repeat instance number of the repeating
#'   event or the repeating instrument. When available in your instance
#'   of REDCap, and passed as NULL, the API will assume a value of 1.
#' @param ... Arguments to be passed to other methods
#' @param error_handling An option for how to handle errors returned by the API.
#'   see \code{\link{redcap_error}}
#' @param config \code{list} Additional configuration parameters to pass to 
#'   \code{\link[httr]{POST}}. These are appended to any parameters in 
#'   \code{rcon$config}.
#' @param api_param \code{list} Additional API parameters to pass into the
#'   body of the API call. This provides users to execute calls with options
#'   that may not otherwise be supported by \code{redcapAPI}.
#' 
#' @author Benjamin Nutter
#'
#' @references
#' Please refer to your institution's API documentation.
#' 
#' Additional details on API parameters are found on the package wiki at
#' \url{https://github.com/vubiostat/redcapAPI/wiki/REDCap-API-Parameters}
#'  
#' @export

deleteFiles <- function(rcon, 
                        record, 
                        field, 
                        event, ...){
  UseMethod("deleteFiles")
}

#' @rdname deleteFiles
#' @export

deleteFiles.redcapApiConnection <- function(rcon, 
                                            record          = NULL, 
                                            field           = NULL, 
                                            event           = NULL, 
                                            repeat_instance = NULL,
                                            ..., 
                                            error_handling  = getOption("redcap_error_handling"),
                                            config          = list(), 
                                            api_param       = list()){
  
  if (is.numeric(record)) record <- as.character(record)
  
  ###########################################################################
  # Check parameters passed to function
  
  coll <- checkmate::makeAssertCollection()
  
  checkmate::assert_class(x = rcon, 
                          classes = "redcapApiConnection", 
                          add = coll)
  
  checkmate::assert_character(x = record, 
                              len = 1, 
                              any.missing = FALSE, 
                              add = coll)
  
  checkmate::assert_character(x = field, 
                              len = 1, 
                              any.missing = FALSE, 
                              add = coll)
  
  checkmate::assert_character(x = event, 
                              len = 1, 
                              any.missing = FALSE, 
                              null.ok = TRUE,
                              add = coll)
  
  checkmate::assert_integerish(x = repeat_instance, 
                               len = 1, 
                               any.missing = FALSE, 
                               null.ok = TRUE,
                               add = coll)
  
  error_handling <- checkmate::matchArg(x = error_handling, 
                                        choices = c("null", "error"), 
                                        .var.name = "error_handling",
                                        add = coll)
  
  checkmate::assert_list(x = config, 
                         names = "named", 
                         add = coll)
  
  checkmate::assert_list(x = api_param, 
                         names = "named", 
                         add = coll)
  
  checkmate::reportAssertions(coll)
  
  # make sure 'field' exist in the project and are 'file' fields
  MetaData <- rcon$metadata()
  
  if (!field %in% MetaData$field_name) 
    coll$push(paste0("'", field, "' does not exist in the project."))
  
  if (MetaData$field_type[MetaData$field_name == field] != "file")
    coll$push(paste0("'", field, "' is not of field type 'file'"))
  
  # make sure 'event' exists in the project
  
  is_project_longitudinal <- as.logical(rcon$projectInformation()$is_longitudinal)
  
  if (is_project_longitudinal)
  {
    EventsList <- rcon$events()
    
    if (nrow(EventsList) == 0)
    {
      message("No events defined in this project. Ignoring the 'event' argument.")
      event <- NULL
    } else {
      checkmate::assert_subset(x = event, 
                               choices = EventsList$unique_event_name, 
                               add = coll)
      checkmate::reportAssertions(coll)
    }
  } else {
    event <- NULL
  }
  
  checkmate::reportAssertions(coll)
  
   ###########################################################################
  # Build the body list
  body <- list(content = 'file',
               action = 'delete', 
               record = record,
               field = field, 
               returnFormat = 'csv', 
               event = event, 
               repeat_instance = repeat_instance)
  
  body <- body[lengths(body) > 0]
  
  #* Delete the file
  #* The tryCatch here seems a little quirky.  My best understanding is that since the API isn't returning
  #* anything into the 'content' attribute returned by POST, POST is casting an error.  Oddly, an error in this
  #* case, an error means the action was successfully performed.  The tryCatch call negotiates that oddity to
  #* get the desired result.
  
  ###########################################################################
  # Make the API Call
  
  response <- makeApiCall(rcon, 
                          body = c(body, api_param), 
                          config = config)
  
  if (response$status_code != "200")
    redcap_error(response, error_handling)
  else 
    message("The file was successfully deleted")
  
}
