#' @name exportPdf
#' @title Export PDF file of Data Collection Instruments (either as blank or with data)
#' 
#' @description This function allows you to download PDF files of data collection 
#' instruments.  The download may be with or without collected data, and may 
#' return a single record, multiple records, or all records.
#'
#' @param rcon A REDCap connection object as generated by \code{redcapConnection}.
#' @param dir The directory into which the file should be saved.
#' @param filename \code{character(1)}. The base of the file name.  If 
#'   \code{record = NULL}, it will be appended with \code{"_blank.pdf"}.  If 
#'   \code{record} has a value, it will be appended with \code{"_record_[record id].pdf"} 
#' @param record The record id for which forms should be downloaded.  May only 
#'   have length 1.
#' @param events The events for which forms should be downloaded
#' @param instruments The instruments for which forms should be downloaded
#' @param all_records Logical. If \code{TRUE} forms for all records are downloaded.
#'   When \code{TRUE}, this overrides the \code{records} argument.
#' @param ... Arguments to be passed to other methods.
#' @param error_handling An option for how to handle errors returned by the API.
#'   see \code{\link{redcap_error}}
#' 
#' @details
#' This function mimics the behavior of "Download PDF of Instruments" button on the
#' REDCap user interface.
#' 
#' @section REDCap API Documentation:
#' This function allows you to export a PDF file for any of the following: 1) a 
#' single data collection instrument (blank), 2) all instruments (blank), 3) a 
#' single instrument (with data from a single record), 4) all instruments (with 
#' data from a single record), or 5) all instruments (with data from ALL records). 
#' This is the exact same PDF file that is downloadable from a project's data entry 
#' form in the web interface, and additionally, the user's privileges with regard to 
#' data exports will be applied here just like they are when downloading the PDF in 
#' the web interface (e.g., if they have de-identified data export rights, then it 
#' will remove data from certain fields in the PDF). If the user has "No Access" data 
#' export rights, they will not be able to use this method, and an error will be returned.
#' 
#' @section REDCap Version:
#' 6.5.0+ 
#' 
#' @section Known REDCap Limitations:
#' None 
#'
#' @return 
#' Creates a PDF file that is saved to the directory specified by the user.
#'
#' @author Benjamin Nutter
#' 
#' @export

exportPdf <- function(rcon, 
                      dir, 
                      filename = "redcap_forms_download", 
                      record = NULL, 
                      events = NULL, 
                      instruments = NULL, 
                      all_records = FALSE, 
                      ...){
  UseMethod("exportPdf")
}

#' @rdname exportPdf
#' @export

exportPdf.redcapApiConnection <- function(rcon, 
                                          dir, 
                                          filename = "redcap_forms_download",
                                          record = NULL, 
                                          events = NULL,
                                          instruments = NULL, 
                                          all_records = FALSE, 
                                          ...,
                                          error_handling = getOption("redcap_error_handling")){
  coll <- checkmate::makeAssertCollection()
  
  checkmate::assert_class(x = rcon,
                          classes = "redcapApiConnection",
                          add = coll)
  
  if (missing(dir)){
    coll$push("'dir' must have a character(1) value")
  }
  else{
    checkmate::assert_character(x = dir,
                                len = 1, 
                                add = coll)
    
    if (is.character(dir)){
      if (!dir.exists(dir)){
        coll$push("'dir' is not an existing directory")
      }
    }
  }
  
  massert(~ filename + record + events + instruments + dir,
          fun = checkmate::assert_character,
          len = list(dir = 1, filename = 1),
          fixed = list(null.ok = TRUE,
                       add = coll))
  
  checkmate::assert_logical(x = all_records,
                            len = 1,
                            add = coll)
  
  error_handling <- checkmate::matchArg(x = error_handling, 
                                        choices = c("null", "error"),
                                        add = coll)
  
  
  checkmate::reportAssertions(coll)
  
  body <- list(token = rcon$token, 
               content = 'pdf', 
               returnFormat = 'csv')
  
  if (!is.null(record)) body[['record']] <- paste0(record, collapse=",")
  if (!is.null(events)) body[['event']] <- paste0(events, collapse=",")
  if (!is.null(instruments)) body[['instrument']] <- paste0(instruments, collapse=",")
  if (all_records) body[['allRecords']] <- 'true'
  
  x <- httr::POST(url = rcon$url,
                  body = body,
                  config = rcon$config)
  
  if (x$status_code != 200) return(redcap_error(x, error_handling))
  
  filename <- 
    if (all_records)
      paste0(filename, "_all_records.pdf")
    else if (is.null(record)) 
      paste0(filename, "_blank.pdf")
    else 
      paste0(filename, "_record_", record, ".pdf")
  
  writeBin(object = as.vector(x$content), 
           con = file.path(dir, filename), 
           useBytes = TRUE)
  
  message("The file was saved to '", file.path(dir, filename), "'")
}
  