% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/hpc-functions.R
\name{.hpc}
\alias{.hpc}
\title{High Precision Calibration}
\usage{
.hpc(mass, minMass, maxMass, hpcCoefficients)
}
\arguments{
\item{mass}{\code{double}, mass calculated traditionally.}

\item{minMass}{\code{double}, lower Threshold for HPC. HPC is only defined
for a range of mass.}

\item{maxMass}{\code{double}, upper Threshold for HPC. HPC is only defined
for a range of mass.}

\item{hpcCoefficients}{\code{doubles}, coefficients needed by the HPC
algorithm (see also:
\code{\link[readBrukerFlexData]{.extractHPCConstants}})}
}
\value{
A vector of HPC corrected mass (\code{double}).
}
\description{
Only basic support (not 100\% identical results) for Bruker Daltonics' HPC.
HPC stands for \bold{H}igh \bold{P}recision \bold{C}alibration.\cr
This is an internal function and should normally not used by the user.
}
\details{
Bruker Daltonics doesn't explain how HPC works. All formula are results of
\dQuote{trial and error}. That is why mass calculated by \code{.hpc}
differs little from original HPC mass.
(In example file 214 of 24860 mass are incorrect;
deviations: min: 6.103515625e-05, max: 0.02935791015625.) \cr
In the manual of mass spectrometry instruments of Bruker Daltonics machines
the *flex series you can find an article about HPC principles: \cr
 Gobom, J. and Mueller, M. and Egelhofer V. and Theiss, D. and
 Lehrach, H. and Nordhoff, E. (2002) \cr
 \dQuote{A Calibration Method That Simplifies and Improves Accurate
  Determination of Peptide Molecular mass by MALDI-TOF MS.},
 \emph{Anal Chem} \bold{74}: 3915-3923 \cr
 \url{http://www.ncbi.nlm.nih.gov/pubmed/12175185}
}
\note{
Please note that .hpc is not correct! You have been warned.
}
\examples{
## load library
library("readBrukerFlexData")

## get examples directory
exampleDirectory <- system.file("Examples", package="readBrukerFlexData")

## read example spectra
## please note: filterZeroIntensities=TRUE is used for compatibility reasons.
##              You should NOT use it!
noHpcSpec <- readBrukerFlexFile(file.path(exampleDirectory,
    "hpc/fid/0_A20/1/1SRef/fid"), filterZeroIntensities=TRUE, useHpc=FALSE)
hpcSpec <- readBrukerFlexFile(file.path(exampleDirectory,
    "hpc/fid/0_A20/1/1SRef/fid"), filterZeroIntensities=TRUE)

## plot spectrum
plot(noHpcSpec$spectrum$mass, noHpcSpec$spectrum$intensity, type="l",
     col="red", xlim=c(1296, 1300))
lines(hpcSpec$spectrum$mass, hpcSpec$spectrum$intensity, type="l",
      col="green", xlim=c(1296, 1300))
legend(x="topright", legend=c("no hpc", "hpc"), col=c("red", "green"), lwd=1)

## show difference between .hpc and original HPC
## load mzXML generated by Bruker Daltonics CompassXport 1.3.5
## you could do it like this:
#library("readMzXmlData")
#cpSpecHpcMzXml <- readMzXmlFile(file.path(exampleDirectory,
#  "hpc/mzXML/hpc.mzXML"))

## or easily use:
data(cpSpecHpcMzXml)

## reduce R double precision to single precision because our CompassXport 1.3.5
## supports only mzXML with precision=32 (only for compatibility reasons)
noHpcSpec$spectrum$mass32 <-
 readBrukerFlexData:::.double2singlePrecision(noHpcSpec$spectrum$mass)
hpcSpec$spectrum$mass32 <-
 readBrukerFlexData:::.double2singlePrecision(hpcSpec$spectrum$mass)

## calculate deviance
d <- noHpcSpec$spectrum$mass32-cpSpecHpcMzXml$spectrum$mass
dHPC <- hpcSpec$spectrum$mass32-cpSpecHpcMzXml$spectrum$mass

## a little summary
cat("without .hpc:\\n",
    "not matching: ", length(cpSpecHpcMzXml$spectrum$mass[d!=0]), " of ",
    length(cpSpecHpcMzXml$spectrum$mass), "; range: ",
    range(abs(d[d!=0])), "\\nwith .hpc:\\n",
    "not matching: ", length(cpSpecHpcMzXml$spectrum$mass[dHPC!=0]), " of ",
    length(cpSpecHpcMzXml$spectrum$mass), "; range: ",
    range(abs(d[dHPC!=0])), "\\n")

##
## doing things manually
##
hpcMass <- readBrukerFlexData:::.hpc(mass=noHpcSpec$spectrum$mass,
 minMass=noHpcSpec$metaData$hpc$limits["minMass"],
 maxMass=noHpcSpec$metaData$hpc$limits["maxMass"],
 hpcCoefficients=noHpcSpec$metaData$hpc$coefficients)

}
\references{
Gobom, J. and Mueller, M. and Egelhofer V. and Theiss, D. and
 Lehrach, H. and Nordhoff, E. (2002) \cr
 \dQuote{A Calibration Method That Simplifies and Improves Accurate
  Determination of Peptide Molecular mass by MALDI-TOF MS.},
 \emph{Anal Chem} \bold{74}: 3915-3923 \cr
 \url{http://www.ncbi.nlm.nih.gov/pubmed/12175185}
}
\seealso{
\code{\link[readBrukerFlexData]{readBrukerFlexDir}},
    \code{\link[readBrukerFlexData]{readBrukerFlexFile}},
    \code{\link[readBrukerFlexData]{.double2singlePrecision}}
}
\keyword{internal}

