#' Process data from the DWD CDC FTP Server
#' 
#' Read climate variables (column meta data) from zip folders downloaded with
#' [dataDWD()].
#' The metadata file `"Metadaten_Parameter.*txt"` in the zip folder `file`
#' is read, processed and returned as a data.frame.\cr
#' `file` can be a vector with several filenames.
#' 
#' @return data.frame of the desired dataset,
#'         or a named list of data.frames if length(file) > 1.
#' @author Berry Boessenkool, \email{berry-b@@gmx.de}, Jun 2018
#' @seealso [dataDWD()], [readDWD()], [`dwdparams`], [newColumnNames()]\cr
#'          [readMeta()] for complete  `Metadaten_Parameter` file.\cr
#'          [website use case](https://bookdown.org/brry/rdwd/use-case-get-all-hourly-rainfall-data-20142016.html#read-the-data)
#' @keywords file
#' @importFrom utils read.table unzip
#' @importFrom berryFunctions checkFile na9 traceCall
#' @importFrom pbapply pblapply
#' @importFrom tools file_path_sans_ext
#' @export
#' @examples
#' # see dataDWD
#' 
#' @param file    Char (vector): name(s) of the file(s) downloaded with [dataDWD()],
#'                e.g. "~/DWDdata/tageswerte_KL_02575_akt.zip"
#' @param params  data.frame: Parameter explanations. DEFAULT: [`dwdparams`]
#' @param quiet   Suppress message about non-abbreviated parameters?
#'                DEFAULT: FALSE through [rdwdquiet()]
#' @param progbar Logical: present a progress bar with estimated remaining time?
#'                If missing and length(file)==1, progbar is internally set to FALSE.
#'                DEFAULT: TRUE
#' 
readVars <- function(
file,
params=dwdparams,
quiet=rdwdquiet(),
progbar=TRUE
)
{
# Optional progress bar:
if(missing(progbar) & length(file)==1) progbar <- FALSE
if(progbar) lapply <- pbapply::pblapply
#
checkFile(file)
#
# loop over each filename
output <- lapply(seq_along(file), function(i)
{
f <- file[i]
# temporary unzipping directory
fn <- tools::file_path_sans_ext(basename(f))
exdir <- paste0(tempdir(),"/", fn)
unzip(f, exdir=exdir)
on.exit(unlink(exdir, recursive=TRUE), add=TRUE)
f <- dir(exdir, pattern="Metadaten_Parameter.*txt", full.names=TRUE)
if(length(f)!=1) return(length(f))
nr <- readLines(f) # number of rows
nr <- sum(!substr(nr, 1, 7) %in% c("Legende", "generie"))
tab <- read.table(f, na.strings=na9(), sep=";", header=TRUE, nrows=nr-1,
                  stringsAsFactors=FALSE)
#
tab <- tab[,c("Parameter", "Parameterbeschreibung", "Einheit")]
tab <- unique(tab)
#
dupli <- duplicated(tab$Parameter)
if(any(dupli)) warning(traceCall(3, "", ": "), "The following entries are",
                       " duplicated: ", toString(unique(tab$Parameter[dupli])),
                       "\nThis occurs in '", fn, "/Metadaten_Parameter*.txt'",
                       call.=FALSE)
rownames(tab) <- NULL
#
# Merge with short variable descriptions:
tab2 <- merge(params, tab, all.y=TRUE)
kurzna <- is.na(tab2$Kurz)
if(any(kurzna) && !quiet) warning(traceCall(3, "", ": "), "The following entries are not",
                        " abbreviated yet: ", toString(tab2$Parameter[kurzna]),
                        "\nThis occurs in '", fn, "/Metadaten_Parameter*.txt'.",
                        "\nPlease inform berry-b@gmx.de so this can be included!\n",
                        call.=FALSE)
#
colnames(tab2)[1] <- "Par"
rownames(tab2) <- tab2$Par
# return column metadata:
return(tab2)
# lapply loop end
})
#
# Warn about zip folders with no meta file:
nometa <- sapply(output, class)=="integer"
if(any(nometa))
 {
 msg <- paste(unlist(output[nometa]), file[nometa], sep=" in ")
 exp <- grepl("_minute", file[nometa]) # expected no meta files
 mexp <- c("\nThis is expected since 1 and 10 minute data do not have ",
           "meta-information in most of the zip folders (as of 2019-02).\n")
 mnexp <- "\nPlease contact berry-b@gmx.de with with a copy of this warning.\n"
 warning(traceCall(1, "", ": "), "The number of determined ",
         "'Metadaten_Parameter*.txt' files should be 1, but is instead:\n",
         paste(msg[ exp],collapse="\n"), if(any( exp)) mexp,
         paste(msg[!exp],collapse="\n"), if(any(!exp)) mnexp,
         call.=FALSE)
 }
#
names(output) <- tools::file_path_sans_ext(basename(file))
output <- if(length(file)==1) output[[1]] else output
return(output)
}



# dwdparams --------------------------------------------------------------------

# updateIndexes will add new rows with NA for Kurz in 'misc/params.txt'
# Copy that to 'misc/params.xlsx', sheet 'input'.
# Add the Kurz entries there manually.
# Copy the 'output' sheet below here.
# Remove the 0 0 row at the end.

#' @title DWD parameter explanations
#' @description Short German parameter explanations for the DWD abbreviations
#' on the CDC FTP server.\cr
#' These are manually created by me and might need to be expanded if the DWD adds
#' more abbreviations.\cr
#' [readVars()] maps them to the variable abbreviations in the
#' `"Metadaten_Parameter.*txt"` file in any given zip folder
#' and will warn about missing entries.
#' 
#' @author Berry Boessenkool, \email{berry-b@@gmx.de}, Jun 2018
#' @seealso [readVars()], [readDWD()]
#' @keywords datasets
#' @export
#' @examples
#' head(dwdparams)
#' 
dwdparams <- unique(read.table(header=TRUE, strip.white=TRUE,
                               stringsAsFactors=FALSE, text="
Parameter	Kurz
ABSF_STD	Absolute_Feuchte
ASH_6	Schneehoehe_Ausstich
ATMO_LBERG	Strahlung_Atmospaere
ATMO_STRAHL	Strahlung_Atmospaere
CD_TER	Wolkendichte
D	Windrichtung
DD	Windrichtung
DD_10	Windrichtung
DK_TER	Windrichtung
DS_10	Strahlung_Himmel_diffus
DX_10	Windrichtung_Maxwind
E_TF_TER	Eisansatz
EK_TER	Erdbodenzustand
F	Windgeschwindigkeit
FD_LBERG	Strahlung_Himmel_diffus
FD_STRAHL	Strahlung_Himmel_diffus
FF	Windgeschwindigkeit
FF_10	Windgeschwindigkeit
FG_LBERG	Strahlung_Global_kurzwellig
FG_STRAHL	Strahlung_Global_kurzwellig
FK_TER	Windstaerke
FM	Windgeschwindigkeit
FMX_10	Windgeschwindigkeit_MaxMean
FNX_10	Windgeschwindigkeit_Min
FX	Windspitze
FX_10	Windgeschwindigkeit_Max
FX_911	Windspitze_Stunde1
FX_911_3	Windspitze_Stunde3
FX_911_6	Windspitze_Stunde6
GEWITTER	Gewitter
GLATTEIS	Glatteis
GRAUPEL	Graupel
GS_10	Strahlung_Global
HAGEL	Hagel
JA_FK	Windstaerke
JA_GEWITTER	Gewitter
JA_GLATTEIS	Glatteis
JA_GRAUPEL	Graupel
JA_HAGEL	Hagel
JA_MX_FX	Windspitze
JA_MX_RS	Niederschlagshoehe_Max
JA_MX_TN	Lufttemperatur_AbsMin
JA_MX_TX	Lufttemperatur_AbsMax
JA_N	Bedeckungsgrad
JA_NEBEL	Nebel
JA_NSH	Schneehoehe_Neu
JA_RR	Niederschlagshoehe
JA_SD_S	Sonnenscheindauer
JA_SH_S	Schneehoehe
JA_STURM_6	Sturm_6Bft
JA_STURM_8	Sturm_8Bft
JA_TAU	Tau
JA_TN	Lufttemperatur_Min
JA_TT	Lufttemperatur
JA_TX	Lufttemperatur_Max
LS_10	Strahlung_langwellig
MO_FK	Windstaerke
MO_GEWITTER	Gewitter
MO_GLATTEIS	Glatteis
MO_GRAUPEL	Graupel
MO_HAGEL	Hagel
MO_N	Bedeckungsgrad
MO_NEBEL	Nebel
MO_NSH	Schneehoehe_Neu
MO_RR	Niederschlagshoehe
MO_SD_S	Sonnenscheindauer
MO_SH_S	Schneehoehe
MO_STURM_6	Sturm_6Bft
MO_STURM_8	Sturm_8Bft
MO_TAU	Tau
MO_TN	Lufttemperatur_Min
MO_TT	Lufttemperatur
MO_TX	Lufttemperatur_Max
MX_FX	Windspitze
MX_RS	Niederschlagshoehe_Max
MX_TN	Lufttemperatur_AbsMin
MX_TX	Lufttemperatur_AbsMax
N_TER	Bedeckungsgrad
NEBEL	Nebel
NM	Bedeckungsgrad
NSH_TAG	Schneehoehe_Neu
P	Luftdruck_NN
P_STD	Luftdruck
P0	Luftdruck_Stationshoehe
PM	Luftdruck
PP_10	Luftdruck
PP_TER	Luftdruck_Terminwert
R1	Niederschlagshoehe
REIF	Reif
RF_10	Relative_Feuchte
RF_STD	Relative_Feuchte
RF_TER	Relative_Feuchte_Terminwert
RF_TU	Relative_Feuchte
RS	Niederschlagshoehe
RS_01	Niederschlagshoehe
RS_IND	Niederschlagsindikator
RS_IND_01	Niederschlagsindikator
RSF	Niederschlagsform
RSK	Niederschlagshoehe
RSKF	Niederschlagsform
RTH_01	Niederschlagshoehe_Tropfen
RWH_01	Niederschlagshoehe_Wippe
RWS_10	Niederschlagshoehe
RWS_DAU_10	Niederschlagsdauer
RWS_IND_10	Niederschlagsindikator
SD_10	Sonnenscheindauer
SD_LBERG	Sonnenscheindauer
SD_SO	Sonnenscheindauer
SD_STRAHL	Sonnenscheindauer
SDK	Sonnenscheindauer
SH_TAG	Schneehoehe
SHK_TAG	Schneehoehe
SLA_10	Windgeschwindigkeit_STABW_lat
SLO_10	Windgeschwindigkeit_STABW_lon
STURM_6	Sturm_6Bft
STURM_8	Sturm_8Bft
TAU	Tau
TD	Taupunkttemperatur
TD_10	Taupunkttemperatur
TD_STD	Taupunkttemperatur
TF_STD	Feuchttemperatur
TF_TER	Feuchttemperatur
TGK	Lufttemperatur_5cm_min
TM5_10	Lufttemperatur_5cm
TMK	Lufttemperatur
TN_10	Lufttemperatur_Min
TN5_10	Lufttemperatur_5cm_min
TNK	Lufttemperatur_Min
TT	Lufttemperatur
TT_10	Lufttemperatur
TT_STD	Lufttemperatur
TT_TER	Lufttemperatur_Terminwert
TT_TU	Lufttemperatur
TX_10	Lufttemperatur_Max
TX5_10	Lufttemperatur_5cm_max
TXK	Lufttemperatur_Max
UPM	Relative_Feuchte
V_N	Bedeckungsgrad
V_S1_CS	Wolkenart_Schicht1
V_S1_CSA	Wolkenart_Abk_Schicht1
V_S1_HHS	Wolkenhoehe_Schicht1
V_S1_NS	Bedeckungsgrad_Schicht1
V_S2_CS	Wolkenart_Schicht2
V_S2_CSA	Wolkenart_Abk_Schicht2
V_S2_HHS	Wolkenhoehe_Schicht2
V_S2_NS	Bedeckungsgrad_Schicht2
V_S3_CS	Wolkenart_Schicht3
V_S3_CSA	Wolkenart_Abk_Schicht3
V_S3_HHS	Wolkenhoehe_Schicht3
V_S3_NS	Bedeckungsgrad_Schicht3
V_S4_CS	Wolkenart_Schicht4
V_S4_CSA	Wolkenart_Abk_Schicht4
V_S4_HHS	Wolkenhoehe_Schicht4
V_S4_NS	Bedeckungsgrad_Schicht4
V_TE002	Erdbodentemperatur_002cm
V_TE002M	Erdbodentemperatur_002cm
V_TE005	Erdbodentemperatur_005cm
V_TE005M	Erdbodentemperatur_005cm
V_TE010	Erdbodentemperatur_010cm
V_TE010M	Erdbodentemperatur_010cm
V_TE020	Erdbodentemperatur_020cm
V_TE020M	Erdbodentemperatur_020cm
V_TE050	Erdbodentemperatur_050cm
V_TE050M	Erdbodentemperatur_050cm
V_TE100	Erdbodentemperatur_100cm
V_VV	Sichtweite
VK_TER	Sichtweite
VP_STD	Dampfdruck
VP_TER	Dampfdruck
VPM	Dampfdruck
WAAS_6	Schneewasseraequivalent
WASH_6	Schneewasseraequivalent_Gesamt
WRTR	Niederschlagsform
WW	Beobachtung
"))
rownames(dwdparams) <- dwdparams$Parameter
