% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rdbw2d_dist.R
\name{rdbw2d.dist}
\alias{rdbw2d.dist}
\title{Bandwidth Selection for Distance-Based RD Designs}
\usage{
rdbw2d.dist(
  Y,
  D,
  b = NULL,
  p = 1,
  kink = c("off", "on"),
  kernel = c("tri", "triangular", "epa", "epanechnikov", "uni", "uniform", "gau",
    "gaussian"),
  bwselect = c("mserd", "imserd", "msetwo", "imsetwo"),
  vce = c("hc1", "hc0", "hc2", "hc3"),
  bwcheck = 20 + p + 1,
  masspoints = c("check", "adjust", "off"),
  C = NULL,
  scaleregul = 1,
  cqt = 0.5
)
}
\arguments{
\item{Y}{Dependent variable; a numeric vector of length \eqn{N}, where \eqn{N} is the sample size.}

\item{D}{Distance-based scores \eqn{\mathbf{D}_i=(\mathbf{D}_{i}(\mathbf{b}_1),\cdots,\mathbf{D}_{i}(\mathbf{b}_J))}; dimension is \eqn{N \times J} where \eqn{N} = sample size and \eqn{J} = number of cutoffs;
non-negative values means data point in treatment group and negative values means data point in control group.}

\item{b}{Optional evaluation points; a matrix or data frame specifying boundary points \eqn{\mathbf{b}_j = (b_{1j}, b_{2j})}, dimension \eqn{J \times 2}.}

\item{p}{Polynomial order for point estimation. Default is \code{p = 1}.}

\item{kink}{Logical; whether to apply kink adjustment. Options: \code{"on"} (default) or \code{"off"}.}

\item{kernel}{Kernel function to use. Options are \code{"unif"}, \code{"uniform"} (uniform), \code{"triag"}, \code{"triangular"} (triangular, default), and \code{"epan"}, \code{"epanechnikov"} (Epanechnikov).}

\item{bwselect}{Bandwidth selection strategy. Options:
\itemize{
\item \code{"mserd"}. One common MSE-optimal bandwidth selector for the boundary RD treatment effect estimator for each evaluation point (default).
\item \code{"imserd"}. IMSE-optimal bandwidth selector for the boundary RD treatment effect estimator based on all evaluation points.
\item \code{"msetwo"}. Two different MSE-optimal bandwidth selectors (control and treatment) for the boundary RD treatment effect estimator for each evaluation point.
\item \code{"imsetwo"}. Two IMSE-optimal bandwidth selectors (control and treatment) for the boundary RD treatment effect estimator based on all evaluation points.
\item \code{"user provided"}. User-provided bandwidths. If \code{h} is not \code{NULL}, then \code{bwselect} is overwritten to \code{"user provided"}.
}}

\item{vce}{Variance-covariance estimator for standard errors.
Options:
\describe{
\item{\code{"hc0"}}{Heteroskedasticity-robust variance estimator without small sample adjustment (White robust).}
\item{\code{"hc1"}}{Heteroskedasticity-robust variance estimator with degrees-of-freedom correction (default).}
\item{\code{"hc2"}}{Heteroskedasticity-robust variance estimator using leverage adjustments.}
\item{\code{"hc3"}}{More conservative heteroskedasticity-robust variance estimator (similar to jackknife correction).}
}}

\item{bwcheck}{If a positive integer is provided, the preliminary bandwidth used in the calculations is enlarged so that at least \code{bwcheck} observations are used. The program stops with “not enough observations” if sample size \eqn{N} < \code{bwcheck}. Default is \code{50 + p + 1}.}

\item{masspoints}{Strategy for handling mass points in the running variable.
Options:
\describe{
\item{\code{"check"}}{Check for repeated values and adjust inference if needed (default).}
\item{\code{"adjust"}}{Adjust bandwidths to guarantee a sufficient number of unique support points.}
\item{\code{"off"}}{Ignore mass points completely.}
}}

\item{C}{Cluster ID variable used for cluster-robust variance estimation with degrees-of-freedom weights.Default is \code{C = NULL}.}

\item{scaleregul}{Scaling factor for the regularization term in bandwidth selection. Default is \code{1}.}

\item{cqt}{Constant controlling subsample fraction for initial bias estimation. Default is \code{0.5}.}
}
\value{
An object of class \code{"rdbw2d.dist"}, containing:
\describe{
\item{\code{bws}}{Data frame of optimal bandwidths for each evaluation point:
\describe{
\item{\code{b1}}{First coordinate of the evaluation point \eqn{b1}.}
\item{\code{b2}}{Second coordinate of the evaluation point \eqn{b2}.}
\item{\code{h0}}{Bandwidth for observations with distance \eqn{D_{i}(\mathbf{b}) < 0}.}
\item{\code{h1}}{Bandwidth for observations with distance \eqn{D_{i}(\mathbf{b}) \geq 0}.}
\item{\code{Nh0}}{Effective sample size for \eqn{D_{i}(\mathbf{b}) < 0}.}
\item{\code{Nh1}}{Effective sample size for \eqn{D_{i}(\mathbf{b}) \geq 0}.}
}
}
\item{\code{mseconsts}}{Data frame of intermediate bias and variance constants used for MSE/IMSE calculations.}
\item{\code{opt}}{List of options used in the function call.}
}
}
\description{
\code{rdbw2d.dist} implements bandwidth selector for distance-based local polynomial boundary regression discontinuity (RD) point estimators with robust bias-corrected pointwise confidence intervals and
uniform confidence bands, developed in Cattaneo, Titiunik and Yu (2025a) with a companion software article Cattaneo, Titiunik and Yu (2025b). For robust bias-correction, see Calonico, Cattaneo, Titiunik (2014).
}
\examples{
set.seed(123)
n <- 5000

# Generate running variables x1 and x2
x1 <- rnorm(n)
x2 <- rnorm(n)

# Define treatment assignment: treated if x1 >= 0
d <- as.numeric(x1 >= 0)

# Generate outcome variable y with some treatment effect
y <- 3 + 2 * x1 + 1.5 * x2 + 1.5 * d + rnorm(n, sd = 0.5)

# Define evaluation points (e.g., at the origin and another point)
eval <- data.frame(x.1 = c(0, 0), x.2 = c(0, 1))

# Compute Euclidean distances to evaluation points
dist.a <- sqrt((x1 - eval$x.1[1])^2 + (x2 - eval$x.2[1])^2)
dist.b <- sqrt((x1 - eval$x.1[2])^2 + (x2 - eval$x.2[2])^2)

# Combine distances into a matrix
D <- as.data.frame(cbind(dist.a, dist.b))

# Assign positive distances for treatment group, negative for control
d_expanded <- matrix(rep(2 * d - 1, times = ncol(D)), nrow = nrow(D), ncol = ncol(D))
D <- D * d_expanded

# Run the rd2d.dist function
bws <- rdbw2d.dist(y, D, b = eval)

# View the estimation results
print(bws)
summary(bws)
}
\references{
\itemize{
\item{\href{https://rdpackages.github.io/references/Cattaneo-Titiunik-Yu_2025_BoundaryRD.pdf}{Cattaneo, M. D., Titiunik, R., Yu, R. R. (2025a).}
Estimation and Inference in Boundary Discontinuity Designs}
\item{\href{https://rdpackages.github.io/references/Cattaneo-Titiunik-Yu_2025_rd2d.pdf}{Cattaneo, M. D., Titiunik, R., Yu, R. R. (2025b).}
rd2d: Causal Inference in Boundary Discontinuity Designs}
\item{\href{https://rdpackages.github.io/references/Calonico-Cattaneo-Titiunik_2014_ECMA.pdf}{Calonico, S., Cattaneo, M. D., Titiunik, R. (2014)}
Robust Nonparametric Confidence Intervals for Regression-Discontinuity Designs}
}
}
\seealso{
\code{\link{rd2d.dist}}, \code{\link{rd2d}}, \code{\link{summary.rdbw2d.dist}}, \code{\link{print.rdbw2d.dist}}
}
\author{
Matias D. Cattaneo, Princeton University. \email{cattaneo@princeton.edu} \cr
Rocío Titiunik, Princeton University. \email{titiunik@princeton.edu} \cr
Ruiqi Rae Yu, Princeton University. \email{rae.yu@princeton.edu}
}
