% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ootb_mblt.R
\name{ootb_mblt}
\alias{ootb_mblt}
\title{Out-of-the-box model-based local thresholding}
\usage{
ootb_mblt(r, z, a, bin = NULL, fix_cs_sky = FALSE)
}
\arguments{
\item{r}{\linkS4class{SpatRaster}. A normalized greyscale image. Typically,
the blue channel extracted from a canopy photograph. Please see
\code{\link{read_caim}} and \code{\link{normalize}}.}

\item{z}{\linkS4class{SpatRaster} built with \code{\link{zenith_image}}.}

\item{a}{\linkS4class{SpatRaster} built with \code{\link{azimuth_image}}.}

\item{bin}{\linkS4class{SpatRaster}. This should be a preliminary
binarization of \code{r} useful for masking pixels that are very likely
to be pure sky pixels.}

\item{fix_cs_sky}{Logical vector of length one. If it is \code{TRUE},
\code{\link{fix_reconstructed_sky}} is used to fix the cone-shaped sky.}
}
\value{
Object from class list containing the binarized image (named
  \emph{bin}) and the reconstructed skies (named \emph{sky_cs} and
  \emph{sky_s}).
}
\description{
Out-of-the-box version of the model-based local thresholding (MBLT)
algorithm.
}
\details{
This function is a hard-coded version of a MBLT pipeline that starts
producing a working binarized image and ends with a refined binarized image.
The pipeline combines these main functions \code{\link{find_sky_pixels}}--if
\code{bin} is \code{NULL}--, \code{\link{fit_coneshaped_model}},
\code{\link{find_sky_pixels_nonnull}}, and \code{\link{fit_trend_surface}}.
The code can be easily inspected by calling \code{ootb_mblt}--no
parenthesis. Advanced users can use that code as a template.

The MBLT algorithm was first presented in
\insertCite{Diaz2018;textual}{rcaiman}. The version presented here differs
from that in the following main aspects:

\itemize{

\item \code{intercept} is set to \code{0}, \code{slope} to \code{1}, and
\code{w} to \code{0.5}

\item This version implements a regional thresholding approach as the first
step instead of a global one. Please refer to \code{\link{find_sky_pixels}}.

\item It does not use asynchronous acquisition under the open sky.
The cone-shaped model (\code{\link{fit_coneshaped_model}}) run without a
filling source and the result of it is used as filling source for trend
surface fitting (\code{\link{fit_trend_surface}}).

\item  \code{\link{find_sky_pixels_nonnull}} is used to update the first
working binarized image, after \code{\link{fit_coneshaped_model}}.

}

This function searches for black objects against a light background. When
regular canopy hemispherical images are provided as input, the algorithm will
find dark canopy elements against a bright sky almost everywhere in the
picture and, therefore, the result will fit user expectations. However, if a
hemispherical photograph taken under the open sky is provided, this algorithm
would be still searching black objects against a light background, so the
darker portions of the sky will be taken as objects, i.e., canopy. As a
consequence, this will not fit users expectations since they are looking for
the classes \emph{Gap} and \emph{No-gap}, no matter if one of those are not
in the picture itself. This kind of error could happen with photographs of
open forests for the same working principle.

If you use this function in your research, please cite
\insertCite{Diaz2018;textual}{rcaiman} in addition to this package.
}
\examples{
\dontrun{
path <- system.file("external/DSCN4500.JPG", package = "rcaiman")
caim <- read_caim(path, c(1280, 960) - 745, 745 * 2, 745 * 2)
z <- zenith_image(ncol(caim), lens("Nikon_FCE9"))
a <- azimuth_image(z)
r <- gbc(caim$Blue)
r[is.na(z)] <- 0 #because FOV > 180
bin <- ootb_mblt(r, z, a)
plot(bin$bin)

ratio <- r/bin$sky_s
ratio <- normalize(ratio, 0, 1, TRUE)
# Alternative 1
plot(apply_thr(ratio, thr_isodata(ratio[!is.na(z)])))

# Alternative 2
g <- sky_grid_segmentation(z, a, 10)
plot(defuzzify(ratio, g))

##Note: In this example, differences are small, but they can be notorious.
}
}
\references{
\insertAllCited{}
}
\seealso{
Other Binarization Functions: 
\code{\link{apply_thr}()},
\code{\link{find_sky_pixels_nonnull}()},
\code{\link{find_sky_pixels}()},
\code{\link{obia}()},
\code{\link{ootb_obia}()},
\code{\link{regional_thresholding}()},
\code{\link{thr_image}()},
\code{\link{thr_isodata}()}
}
\concept{Binarization Functions}
