% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/chunk.R
\name{ragnar_chunk}
\alias{ragnar_chunk}
\alias{ragnar_segment}
\alias{ragnar_chunk_segments}
\title{Chunk text}
\usage{
ragnar_chunk(
  x,
  max_size = 1600L,
  boundaries = c("paragraph", "sentence", "line_break", "word", "character"),
  ...,
  trim = TRUE,
  simplify = TRUE
)

ragnar_segment(x, boundaries = "sentence", ..., trim = FALSE, simplify = TRUE)

ragnar_chunk_segments(x, max_size = 1600L, ..., simplify = TRUE, trim = TRUE)
}
\arguments{
\item{x}{A character vector, list of character vectors, or data frame containing a \code{text} column.}

\item{max_size}{Integer. The maximum number of characters in each chunk.
Defaults to \code{1600}, which typically is approximately 400 tokens, or 1 page of text.}

\item{boundaries}{A sequence of boundary types to use in order until
\code{max_size} is satisfied. Valid values are \code{"sentence"}, \code{"word"},
\code{"line_break"}, \code{"character"}, \code{"paragraph"}, or a \code{stringr_pattern} object
like \code{stringr::fixed()}.}

\item{...}{Additional arguments passed to internal functions.
tokenizer to use \code{tokens} instead of characters as the count (not fully
implemented yet)}

\item{trim}{logical, whether to trim leading and trailing whitespace from
strings. Default \code{TRUE}.}

\item{simplify}{Logical. If \code{TRUE}, the output is simplified. If \code{FALSE},
returns a vector that has the same length as \code{x}. If \code{TRUE}, character
strings are \code{unlist()}ed, and dataframes are \code{tidyr::unchop()}ed.}
}
\value{
\itemize{
\item For character input with \code{simplify = FALSE}: A list of character vectors
\item For character input with \code{simplify = TRUE}: A character vector of chunks
\item For data frame input with \code{simplify = FALSE}: A data frame with the same number of rows as the input, where the
\code{text} column transformed into a list of chararacter vectors.
\item For data frame input with \code{simplify = TRUE}: Same as a data frame input with \code{simplify=FALSE}, with the
\code{text} column expanded by \code{tidyr::unchop()}
}
}
\description{
Functions for chunking text into smaller pieces while preserving meaningful
semantics. These functions provide flexible ways to split text based on
various boundaries (sentences, words, etc.) while controlling chunk sizes and
overlap.
}
\details{
Chunking is the combination of two fundamental operations:
\itemize{
\item identifying boundaries: finding character positions where it makes sense to split a string.
\item extracting slices: extracting substrings using the candidate boundaries to produce chunks that
match the requested \code{chunk_size} and \code{chunk_overlap}
}

\code{ragnar_chunk()} is a higher-level function that does both, identifies boundaries and extracts slices.

If you need lower-level control, you can alternatively use the lower-level functions
\code{ragnar_segment()} in combination with \code{ragnar_chunk_segments()}.

\code{ragnar_segment()}: Splits text at semantic boundaries.

\code{ragnar_chunk_segments()}: Combines text segments into chunks.

For most usecases, these two are equivalent:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{x |> ragnar_chunk()
x |> ragnar_segment() |> ragnar_chunk_segments()
}\if{html}{\out{</div>}}

When working with data frames, these functions preserve all columns and use
\code{tidyr::unchop()} to handle the resulting list-columns when \code{simplify = TRUE}.
}
\examples{
# Basic chunking with max size
text <- "This is a long piece of text. It has multiple sentences.
         We want to split it into chunks. Here's another sentence."
ragnar_chunk(text, max_size = 40) # splits at sentences

# smaller chunk size: first splits at sentence boundaries, then word boundaries
ragnar_chunk(text, max_size = 20)

# only split at sentence boundaries. Note, some chunks are oversized
ragnar_chunk(text, max_size = 20, boundaries = c("sentence"))

# only consider word boundaries when splitting:
ragnar_chunk(text, max_size = 20, boundaries = c("word"))

# first split at sentence boundaries, then word boundaries,
# as needed to satisfy `max_chunk`
ragnar_chunk(text, max_size = 20, boundaries = c("sentence", "word"))

# Use a stringr pattern to find semantic boundaries
ragnar_chunk(text, max_size = 10, boundaries = stringr::fixed(". "))
ragnar_chunk(text, max_size = 10, boundaries = list(stringr::fixed(". "), "word"))


# Working with data frames
df <- data.frame(
  id = 1:2,
  text = c("First sentence. Second sentence.", "Another sentence here.")
)
ragnar_chunk(df, max_size = 20, boundaries = "sentence")
ragnar_chunk(df$text, max_size = 20, boundaries = "sentence")

# Chunking pre-segmented text
segments <- c("First segment. ", "Second segment. ", "Third segment. ", "Fourth segment. ")
ragnar_chunk_segments(segments, max_size = 20)
ragnar_chunk_segments(segments, max_size = 40)
ragnar_chunk_segments(segments, max_size = 60)

}
