% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mqgam.R
\name{mqgam}
\alias{mqgam}
\title{Fit multiple smooth additive quantile regression models}
\usage{
mqgam(form, data, qu, lsig = NULL, err = 0.05,
  multicore = !is.null(cluster), cluster = NULL, ncores = detectCores() -
  1, paropts = list(), control = list(), argGam = NULL)
}
\arguments{
\item{form}{A GAM formula, or a list of formulae. See ?mgcv::gam details.}

\item{data}{A data frame or list containing the model response variable and covariates required by the formula.
By default the variables are taken from environment(formula): typically the environment from which gam is called.}

\item{qu}{A vectors of quantiles of interest. Each entry should be in (0, 1).}

\item{lsig}{The value of the log learning rate used to create the Gibbs posterior. By defauls \code{lsig=NULL} and this
parameter is estimated by posterior calibration described in Fasiolo et al. (2016). Obviously, the function is much faster
if the user provides a value.}

\item{err}{An upper bound on the error of the estimated quantile curve. Should be in (0, 1). If it is a vector, it should be of the 
same length of \code{qu}. See Fasiolo et al. (2016) for details.}

\item{multicore}{If TRUE the calibration will happen in parallel.}

\item{cluster}{An object of class \code{c("SOCKcluster", "cluster")}. This allowes the user to pass her own cluster,
which will be used if \code{multicore == TRUE}. The user has to remember to stop the cluster.}

\item{ncores}{Number of cores used. Relevant if \code{multicore == TRUE}.}

\item{paropts}{a list of additional options passed into the foreach function when parallel computation is enabled. 
This is important if (for example) your code relies on external data or packages: 
use the .export and .packages arguments to supply them so that all cluster nodes 
have the correct environment set up for computing.}

\item{control}{A list of control parameters. The only one relevant here is \code{link}, which is the link function
used (see \code{?elf} and \code{?elflss} for defaults). All other control parameters are used by 
\code{tuneLearnFast}. See \code{?tuneLearnFast} for details.}

\item{argGam}{A list of parameters to be passed to \code{mgcv::gam}. This list can potentially include all the arguments listed
in \code{?gam}, with the exception of \code{formula}, \code{family} and \code{data}.}
}
\value{
A list with entries: \itemize{
                  \item{\code{fit} = a \code{gamObject}, one for each entry of \code{qu}.  Notice that the
                                     slots \code{model} and \code{smooth} of each object has been removed to save memory. 
                                     See \code{?gamObject}. }
                  \item{\code{model} = the \code{model} slot of the \code{gamObject}s in the \code{fit} slot. This is the same for every
                                       fit, hence only one copy is stored.}
                  \item{\code{smooth} = the \code{smooth} slot of the \code{gamObject}s in the \code{fit} slot. This is the same for every
                                       fit, hence only one copy is stored.}
                  \item{\code{calibr} = a list which is the output of an internal call to \code{tuneLearnFast}, which is used for calibrating
                                        the learning rate. See \code{?tuneLearnFast} for details.}
}
}
\description{
This function fits a smooth additive regression model to several quantiles.
}
\examples{

#####
# Multivariate Gaussian example
####
library(qgam)
set.seed(2)
dat <- gamSim(1, n=300, dist="normal", scale=2)

fit <- mqgam(y~s(x0)+s(x1)+s(x2)+s(x3), data=dat, err = 0.05, qu = c(0.2, 0.8))

invisible( qdo(fit, 0.2, plot, pages = 1) )

#####
# Univariate "car" example
####
\dontrun{
library(qgam); library(MASS)

# Fit for quantile 0.8 using the best sigma
quSeq <- c(0.2, 0.4, 0.6, 0.8)
set.seed(6436)
fit <- mqgam(accel~s(times, k=20, bs="ad"), data = mcycle, err = 0.05, qu = quSeq)

# Plot the fit
xSeq <- data.frame(cbind("accel" = rep(0, 1e3), "times" = seq(2, 58, length.out = 1e3)))
plot(mcycle$times, mcycle$accel, xlab = "Times", ylab = "Acceleration", ylim = c(-150, 80))
for(iq in quSeq){
  pred <- qdo(fit, iq, predict, newdata = xSeq)
  lines(xSeq$times, pred, col = 2)
}
}

}
\references{
Fasiolo, M., Goude, Y., Nedellec, R. and Wood, S. N. (2017). Fast calibrated additive quantile regression. Available at
            \url{https://arxiv.org/abs/1707.03307}.
}
\author{
Matteo Fasiolo <matteo.fasiolo@gmail.com>.
}
