#' Write an agent's validation plan to a YAML file
#' 
#' @description 
#' With `agent_yaml_write()` we can take an existing *agent* and write that
#' *agent*'s validation plan to a YAML file. With **pointblank** YAML, we can
#' modify the YAML markup if so desired, or, use as is to create a new agent
#' with the [agent_yaml_read()] function. That *agent* will have a validation
#' plan and is ready to [interrogate()] the data. We can go a step further and
#' perform an interrogation directly from the YAML file with the
#' [agent_yaml_interrogate()] function. That returns an agent with intel (having
#' already interrogated the target data table).
#'
#' One requirement for writing the *agent* to YAML is that we need to have
#' table-reading function (`read_fn`) specified (it's a function that is used to
#' read the target table when [interrogate()] is called). This option can be set
#' when using [create_agent()] or with [set_read_fn()] (for use with an existing
#' *agent*).
#' 
#' @param agent An *agent* object of class `ptblank_agent` that is created with
#'   [create_agent()].
#' @param filename The name of the YAML file to create on disk. It is
#'   recommended that either the `.yaml` or `.yml` extension be used for this
#'   file.
#' @param path An optional path to which the YAML file should be saved (combined
#'   with `filename`).
#'   
#' @examples
#' # Let's go through the process of
#' # developing an agent with a validation
#' # plan (to be used for the data quality
#' # analysis of the `small_table` dataset),
#' # and then offloading that validation
#' # plan to a pointblank YAML file
#' 
#' # We ought to think about what's
#' # tolerable in terms of data quality so
#' # let's designate proportional failure
#' # thresholds to the `warn`, `stop`, and
#' # `notify` states using `action_levels()`
#' al <- 
#'   action_levels(
#'     warn_at = 0.10,
#'     stop_at = 0.25,
#'     notify_at = 0.35
#'   )
#' 
#' # Now create a pointblank `agent` object
#' # and give it the `al` object (which
#' # serves as a default for all validation
#' # steps which can be overridden); the
#' # data will be referenced in a `read_fn`
#' # (a requirement for writing to YAML)
#' agent <- 
#'   create_agent(
#'     read_fn = ~small_table,
#'     name = "example",
#'     actions = al
#'   )
#' 
#' # Then, as with any `agent` object, we
#' # can add steps to the validation plan by
#' # using as many validation functions as we
#' # want
#' agent <-
#'   agent %>% 
#'   col_exists(vars(date, date_time)) %>%
#'   col_vals_regex(
#'     vars(b), "[0-9]-[a-z]{3}-[0-9]{3}"
#'   ) %>%
#'   rows_distinct() %>%
#'   col_vals_gt(vars(d), 100) %>%
#'   col_vals_lte(vars(c), 5)
#'
#' # The agent can be written to a pointblank
#' # YAML file with `agent_yaml_write()`
#' # agent_yaml_write(agent, filename = "x.yml")
#' 
#' # The 'x.yml' file is available in the package
#' # through `system.file()`
#' yml_file <- 
#'   system.file("x.yml", package = "pointblank")
#' 
#' # We can view the YAML file in the console
#' # with the `agent_yaml_string()` function
#' agent_yaml_string(path = yml_file)
#' 
#' # The YAML can also be printed in the console
#' # by supplying the agent as the input
#' agent_yaml_string(agent = agent)
#' 
#' # At a later time, the YAML file can
#' # be read into a new agent with the
#' # `agent_yaml_read()` function
#' agent <- agent_yaml_read(path = yml_file)
#' 
#' class(agent)
#' 
#' # We can interrogate the data (which
#' # is accessible through the `read_fn`)
#' # with `interrogate()` and get an
#' # agent with intel, or, we can
#' # interrogate directly from the YAML
#' # file with `agent_yaml_interrogate()`
#' agent <- 
#'   agent_yaml_interrogate(path = yml_file)
#' 
#' class(agent)
#' 
#' @family pointblank YAML
#' @section Function ID:
#' 7-1
#' 
#' @export
agent_yaml_write <- function(agent,
                             filename,
                             path = NULL) {
  
  if (!is.null(path)) {
    filename <- file.path(path, filename)
  }
  
  filename <- fs::path_expand(filename)
  
  as_agent_yaml_list(agent) %>%
    yaml::write_yaml(
      file = filename,
      handlers = list(
        logical = function(x) {
          result <- ifelse(x, "true", "false")
          class(result) <- "verbatim"
          result
        }
      )
    )
}

#' Display **pointblank** YAML using an agent or a YAML file
#' 
#' With **pointblank** YAML, we can serialize an agent's validation plan (with
#' [agent_yaml_write()]), read it back later with a new agent (with
#' [agent_yaml_read()]), or perform an interrogation on the target data table
#' directly with the YAML file (with [agent_yaml_interrogate()]). The
#' `agent_yaml_string()` function allows us to inspect the YAML generated by
#' [agent_yaml_write()] in the console, giving us a look at the YAML without
#' needing to open the file directly. Alternatively, we can provide an *agent*
#' to the `agent_yaml_string()` and view the YAML representation of the
#' validation plan without needing to write the YAML to disk beforehand.
#'
#' @param agent An *agent* object of class `ptblank_agent` that is created with
#'   [create_agent()].
#' @param path A path to a YAML file that specifies a validation plan for an
#'   *agent*.
#'   
#' @examples 
#' # Let's create a validation plan for the
#' # data quality analysis of the `small_table`
#' # dataset; we need an agent and its
#' # table-reading function enables retrieval
#' # of the target table
#' agent <- 
#'   create_agent(
#'     read_fn = ~small_table,
#'     name = "example",
#'     actions = action_levels(
#'       warn_at = 0.10,
#'       stop_at = 0.25,
#'       notify_at = 0.35
#'     )
#'   ) %>%
#'   col_exists(vars(date, date_time)) %>%
#'   col_vals_regex(
#'     vars(b), "[0-9]-[a-z]{3}-[0-9]{3}"
#'   ) %>%
#'   rows_distinct() %>%
#'   col_vals_gt(vars(d), 100) %>%
#'   col_vals_lte(vars(c), 5)
#'
#' # We can view the YAML file in the console
#' # with the `agent_yaml_string()` function,
#' # providing the `agent` object as the input
#' agent_yaml_string(agent = agent)
#'
#' # The agent can be written to a pointblank
#' # YAML file with `agent_yaml_write()`
#' # agent_yaml_write(agent, filename = "x.yml")
#' 
#' # The 'x.yml' file is available in the package
#' # through `system.file()`
#' yml_file <- 
#'   system.file("x.yml", package = "pointblank")
#' 
#' # The `agent_yaml_string()` function can
#' # be used with the YAML file as well
#' agent_yaml_string(path = yml_file)
#' 
#' # At a later time, the YAML file can
#' # be read into a new agent with the
#' # `agent_yaml_read()` function
#' agent <- agent_yaml_read(path = yml_file)
#' 
#' class(agent)
#'   
#' @family pointblank YAML
#' @section Function ID:
#' 7-4
#' 
#' @export
agent_yaml_string <- function(agent = NULL,
                              path = NULL) {
  
  if (is.null(agent) && is.null(path)) {
    stop(
      "An `agent` object or a `path` to a YAML file must be specified.",
      call. = FALSE
    )
  }
  
  if (!is.null(agent) && !is.null(path)) {
    stop("Only one of `agent` or `path` should be specified.", call. = FALSE)
  }
  
  if (!is.null(agent)) {
    
    message(
      as_agent_yaml_list(agent) %>%
        yaml::as.yaml(
          handlers = list(
            logical = function(x) {
              result <- ifelse(x, "true", "false")
              class(result) <- "verbatim"
              result
            }
          )
        )
    )
    
  } else {
    message(readLines(path) %>% paste(collapse = "\n"))
  }
}

as_vars_fn <- function(columns) {
  paste0("vars(", columns,")")
}

as_list_preconditions <- function(preconditions) {
  if (is.null(preconditions[[1]])) {
    return(NULL)
  } else {
    return(as.character(preconditions))
  }
}

to_list_action_levels <- function(actions) {
  
  agent_actions <- actions
  agent_actions[sapply(agent_actions, is.null)] <- NULL
  agent_actions$fns[sapply(agent_actions$fns, is.null)] <- NULL
  
  if (length(agent_actions$fns) == 0) agent_actions$fns <- NULL
  
  if (length(agent_actions$fns) == 0) {
    agent_actions$fns <- NULL
  } else {
    agent_actions$fns <-
      lapply(
        agent_actions$fns,
        FUN = function(x) {
          if (!is.null(x)) x %>% as.character() %>% paste(collapse = "")
        }
      )
  }
  
  list(actions = agent_actions)
}

as_action_levels <- function(actions, actions_default = NULL) {

  agent_actions <- actions
  agent_actions[sapply(agent_actions, is.null)] <- NULL
  agent_actions$fns[sapply(agent_actions$fns, is.null)] <- NULL
  
  if (length(agent_actions$fns) == 0) {
    agent_actions$fns <- NULL
  } else {
    agent_actions$fns <-
      lapply(
        agent_actions$fns,
        FUN = function(x) {
          if (!is.null(x)) x %>% as.character() %>% paste(collapse = "")
        }
      )
  }

  if (!is.null(actions_default)) {
    if (identical(agent_actions, actions_default)) {
      return(NULL)
    }
  }
  
  agent_actions
}

get_schema_list <- function(schema) {

  vals <- schema
  
  complete <- schema$`__complete__`
  in_order <- schema$`__in_order__`
  
  type <- ifelse(inherits(schema, "r_type"), "r", "sql")
  
  vals <- vals[!(names(vals) %in% c("__complete__", "__in_order__"))]
  
  if (type == "sql") {
    vals <- c(vals, list(`.db_col_types` = "sql"))
  }

  list(
    schema = vals, 
    complete = complete,
    in_order = in_order
  )
}

to_list_read_fn <- function(read_fn) {
  list(read_fn = read_fn %>% rlang::as_label())
}

to_list_name <- function(name) {
  list(name = name)
}

get_arg_value <- function(value) {
  
  if (inherits(value, "list") && inherits(value[[1]], "quosures")) {
    out <- paste0("vars(", rlang::as_label(value[[1]][[1]]),")")
  } else if (inherits(value, "list") && inherits(value[[1]], "numeric")) {
    out <- value[[1]] 
  } else {
    out <- as.character(value[[1]])
  }
  
  out
}

get_arg_value_lr <- function(value) {

  if (inherits(value, "quosure")) {
    out <- paste0("vars(", rlang::as_label(value),")")
  } else if (inherits(value, "numeric")) {
    out <- value
  } else {
    out <- as.character(value)
  }
  
  out
}

prune_lst_step <- function(lst_step) {
  
  if ("preconditions" %in% names(lst_step[[1]]) &&
      is.null(lst_step[[1]][["preconditions"]])) {
    lst_step[[1]]["preconditions"] <- NULL
  }
  if ("na_pass" %in% names(lst_step[[1]]) &&
      !lst_step[[1]][["na_pass"]]) {
    lst_step[[1]]["na_pass"] <- NULL
  }
  if ("active" %in% names(lst_step[[1]]) &&
      lst_step[[1]][["active"]]) {
    lst_step[[1]]["active"] <- NULL
  }
  if ("complete" %in% names(lst_step[[1]]) &&
      lst_step[[1]][["complete"]]) {
    lst_step[[1]]["complete"] <- NULL
  }
  if ("in_order" %in% names(lst_step[[1]]) &&
      lst_step[[1]][["in_order"]]) {
    lst_step[[1]]["in_order"] <- NULL
  }
  if ("actions" %in% names(lst_step[[1]])) {
    if (length(lst_step[[1]][["actions"]][["action_levels"]]) == 1 &&
        length(lst_step[[1]][["actions"]][["action_levels"]][["fns"]]) == 0) {
      
      lst_step[[1]][["actions"]] <- NULL
    } else if (is.null(lst_step[[1]][["actions"]])) {
      lst_step[[1]][["actions"]] <- NULL
    }
  }
  
  lst_step
}

as_agent_yaml_list <- function(agent) {

  if (is.null(agent$read_fn)) {
    stop(
      "The agent must have a `read_fn` value to transform it into YAML.",
       call. = FALSE
    )
  }

  action_levels_default <- as_action_levels(agent$actions)
  end_fns <- agent$end_fns %>% unlist()
  
  lst_name <- to_list_name(agent$name)
  lst_read_fn <- to_list_read_fn(agent$read_fn)
  
  if (is.null(action_levels_default)) {
    lst_action_levels <- NULL
  } else {
    lst_action_levels <- list(actions = action_levels_default)
  }
  
  if (is.null(end_fns)) {
    lst_end_fns <- NULL
  } else {
    lst_end_fns <- list(end_fns = as.character(end_fns))
  }
  
  if (is.null(agent$embed_report) || 
      (!is.null(agent$embed_report) && !agent$embed_report)) {
    lst_embed_report <- NULL
  } else {
    lst_embed_report <- list(embed_report = agent$embed_report)
  }

  if (is.null(agent$reporting_lang) || 
      (!is.null(agent$reporting_lang) && agent$reporting_lang == "en")) {
    lst_reporting_lang <- NULL
  } else {
    lst_reporting_lang <- list(reporting_lang = agent$reporting_lang)
  }

  # Select only the necessary columns from the agent's `validation_set` 
  agent_validation_set <- 
    agent$validation_set %>% 
    dplyr::select(
      assertion_type, column, values, na_pass,
      preconditions, actions, brief, active
    )
  
  all_steps <- list()
  
  for (i in seq_len(nrow(agent_validation_set))) {
    
    step_list <- agent_validation_set[i, ] %>% as.list()
    
    validation_fn <- step_list$assertion_type
    
    if (validation_fn %in% c(
      "col_vals_lt", "col_vals_lte",
      "col_vals_equal", "col_vals_not_equal",
      "col_vals_gte", "col_vals_gt"
    )) {
      
      lst_step <- 
        list(
          validation_fn = list(
            columns = as_vars_fn(step_list$column[[1]]),
            value = get_arg_value(step_list$values),
            na_pass = step_list$na_pass,
            preconditions = as_list_preconditions(step_list$preconditions),
            actions = as_action_levels(step_list$actions[[1]], action_levels_default),
            active = step_list$active
          )
        )
      
    } else if (validation_fn %in% c("col_vals_between", "col_vals_not_between")) {

      lst_step <- 
        list(
          validation_fn = list(
            columns = as_vars_fn(step_list$column[[1]]),
            left = get_arg_value_lr(step_list$values[[1]][[1]]),
            right = get_arg_value_lr(step_list$values[[1]][[2]]),
            inclusive = as.logical(c(names(step_list$values[[1]][1]), names(step_list$values[[1]][1]))),
            na_pass = step_list$na_pass,
            preconditions = as_list_preconditions(step_list$preconditions),
            actions = as_action_levels(step_list$actions[[1]], action_levels_default),
            active = step_list$active
          )
        )
      
    } else if (validation_fn %in% c("col_vals_in_set", "col_vals_not_in_set")) {

      lst_step <- 
        list(
          validation_fn = list(
            columns = as_vars_fn(step_list$column[[1]]),
            set = step_list$values[[1]],
            preconditions = as_list_preconditions(step_list$preconditions),
            actions = as_action_levels(step_list$actions[[1]], action_levels_default),
            active = step_list$active
          )
        )

    } else if (validation_fn %in% c("col_vals_null", "col_vals_not_null")) {
      
      lst_step <- 
        list(
          validation_fn = list(
            columns = as_vars_fn(step_list$column[[1]]),
            preconditions = as_list_preconditions(step_list$preconditions),
            actions = as_action_levels(step_list$actions[[1]], action_levels_default),
            active = step_list$active
          )
        )
      
    } else if (validation_fn == "col_vals_regex") {

      lst_step <- 
        list(
          validation_fn = list(
            columns = as_vars_fn(step_list$column[[1]]),
            regex = get_arg_value(step_list$values),
            preconditions = as_list_preconditions(step_list$preconditions),
            actions = as_action_levels(step_list$actions[[1]], action_levels_default),
            active = step_list$active
          )
        )
      
    } else if (grepl("col_is_", validation_fn) || validation_fn == "col_exists") {

      lst_step <- 
        list(
          validation_fn = list(
            columns = as_vars_fn(step_list$column[[1]]),
            actions = as_action_levels(step_list$actions[[1]], action_levels_default),
            active = step_list$active
          )
        )
      
    } else if (validation_fn == "col_vals_expr") {

      lst_step <- 
        list(
          validation_fn = list(
            expr = paste0("~", rlang::as_label(step_list$values[[1]])),
            preconditions = as_list_preconditions(step_list$preconditions),
            actions = as_action_levels(step_list$actions[[1]], action_levels_default),
            active = step_list$active
          )
        )
      
    } else if (validation_fn == "rows_distinct") {

      if (is.na(step_list$column[[1]][[1]])) {
        vars_cols <- NULL
      } else {
        vars_cols <- as_vars_fn(step_list$column[[1]])
      }
      
      lst_step <- 
        list(
          validation_fn = list(
            columns = vars_cols,
            preconditions = as_list_preconditions(step_list$preconditions),
            actions = as_action_levels(step_list$actions[[1]], action_levels_default),
            active = step_list$active
          )
        )
    
    } else if (validation_fn == "col_schema_match") {
      
      schema_list <- get_schema_list(schema = step_list$values[[1]])
      
      lst_step <- 
        list(
          validation_fn = list(
            schema = schema_list$schema,
            complete = schema_list$complete,
            in_order = schema_list$in_order,
            actions = as_action_levels(step_list$actions[[1]], action_levels_default),
            active = step_list$active
          )
        )
      
    } else if (validation_fn == "conjointly") {
      
      lst_step <- 
        list(
          validation_fn = list(
            fns = as.character(step_list$values[[1]]),
            preconditions = as_list_preconditions(step_list$preconditions),
            actions = as_action_levels(step_list$actions[[1]], action_levels_default),
            active = step_list$active
          )
        )
    }

    # Remove list elements that are representative of defaults
    lst_step <- prune_lst_step(lst_step)

    # Set the top level list-element name to that of
    # the validation function
    names(lst_step) <- validation_fn
    all_steps <- c(all_steps, list(lst_step))
  }
  
  c(
    lst_name,
    lst_read_fn,
    lst_action_levels,
    lst_end_fns,
    lst_embed_report,
    lst_reporting_lang,
    list(steps = all_steps)
  )
}
