% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/yaml_read_agent.R
\name{yaml_read_agent}
\alias{yaml_read_agent}
\title{Read a \strong{pointblank} YAML file to create an \emph{agent} object}
\usage{
yaml_read_agent(filename, path = NULL)
}
\arguments{
\item{filename}{The name of the YAML file that contains fields related to an
\emph{agent}.}

\item{path}{An optional path to the YAML file (combined with \code{filename}).}
}
\value{
A \code{ptblank_agent} object.
}
\description{
With \code{yaml_read_agent()} we can read a \strong{pointblank} YAML file that
describes a validation plan to be carried out by an \emph{agent} (typically
generated by the \code{\link[=yaml_write]{yaml_write()}} function. What's returned is a new \emph{agent}
with that validation plan, ready to interrogate the target table at will
(using the table-prep formula that is set with the \code{tbl} argument of
\code{\link[=create_agent]{create_agent()}}). The agent can be given more validation steps if needed
before using \code{\link[=interrogate]{interrogate()}} or taking part in any other agent ops (e.g.,
writing to disk with outputs intact via \code{\link[=x_write_disk]{x_write_disk()}} or again to
\strong{pointblank} YAML with \code{\link[=yaml_write]{yaml_write()}}).

To get a picture of how \code{yaml_read_agent()} is interpreting the validation
plan specified in the \strong{pointblank} YAML, we can use the
\code{\link[=yaml_agent_show_exprs]{yaml_agent_show_exprs()}} function. That function shows us (in the console)
the \strong{pointblank} expressions for generating the described validation plan.
}
\section{Function ID}{

11-2
}

\examples{
if (interactive()) {

# Let's go through the process of
# developing an agent with a validation
# plan (to be used for the data quality
# analysis of the `small_table` dataset),
# and then offloading that validation
# plan to a pointblank YAML file; this
# will be read in with `yaml_read_agent()`

# Creating an `action_levels` object is a
# common workflow step when creating a
# pointblank agent; we designate failure
# thresholds to the `warn`, `stop`, and
# `notify` states using `action_levels()`
al <- 
  action_levels(
    warn_at = 0.10,
    stop_at = 0.25,
    notify_at = 0.35
  )

# Now create a pointblank `agent` object
# and give it the `al` object (which
# serves as a default for all validation
# steps which can be overridden); the
# data will be referenced in `tbl` with
# a table-prep formula (a requirement
# for writing to YAML)
agent <- 
  create_agent(
    tbl = ~ small_table,
    tbl_name = "small_table",
    label = "A simple example with the `small_table`.",
    actions = al
  )

# Then, as with any `agent` object, we
# can add steps to the validation plan by
# using as many validation functions as we
# want
agent <-
  agent \%>\% 
  col_exists(vars(date, date_time)) \%>\%
  col_vals_regex(
    vars(b),
    regex = "[0-9]-[a-z]{3}-[0-9]{3}"
  ) \%>\%
  rows_distinct() \%>\%
  col_vals_gt(vars(d), value = 100) \%>\%
  col_vals_lte(vars(c), value = 5)

# The agent can be written to a pointblank
# YAML file with `yaml_write()`
yaml_write(
  agent = agent,
  filename = "agent-small_table.yml"
)

# The 'agent-small_table.yml' file is
# available in the package through
# `system.file()`
yml_file <- 
  system.file(
    "yaml", "agent-small_table.yml",
    package = "pointblank"
  )

# We can view the YAML file in the console
# with the `yaml_agent_string()` function
yaml_agent_string(filename = yml_file)

# The YAML can also be printed in the console
# by supplying the agent as the input
yaml_agent_string(agent = agent)

# At some later time, the YAML file can
# be read as a new agent with the
# `yaml_read_agent()` function
agent <- yaml_read_agent(filename = yml_file)

class(agent)

# We can interrogate the data (which
# is accessible through the table-prep
# formula supplied initially) with
# `interrogate()` and get an agent
# with intel, or, we can interrogate
# directly from the YAML file with
# `yaml_agent_interrogate()`
agent <- 
  yaml_agent_interrogate(
    filename = yml_file
  )

class(agent)

}

}
\seealso{
Other pointblank YAML: 
\code{\link{yaml_agent_interrogate}()},
\code{\link{yaml_agent_show_exprs}()},
\code{\link{yaml_agent_string}()},
\code{\link{yaml_exec}()},
\code{\link{yaml_informant_incorporate}()},
\code{\link{yaml_read_informant}()},
\code{\link{yaml_write}()}
}
\concept{pointblank YAML}
