% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Generator.R
\name{Generator}
\alias{Generator}
\title{R6 class representing a dynamic attribute generator}
\description{
\code{\link[R6:R6Class]{R6}} class representing a model that dynamically generates
attribute values (\emph{outputs}) via reading data from files, running assigned
functions, generating sample distributions, or built-in functions (assigned as
\emph{default} in inherited classes), using simulation sample parameters
(\emph{inputs}).
}
\examples{
# U Island example region
coordinates <- data.frame(
  x = rep(seq(177.01, 177.05, 0.01), 5),
  y = rep(seq(-18.01, -18.05, -0.01), each = 5)
)
coordinates <- coordinates[c(7, 9, 12, 14, 17:19), ]
region <- Region$new(coordinates = coordinates, use_raster = FALSE)
# Spatial correlation
spatial_correlation <- SpatialCorrelation$new(
  region = region, correlation_amplitude = 0.6,
  correlation_breadth = 300
)
spatial_correlation$calculate_compact_decomposition(decimals = 4)
# Example habitat suitability in file
saveRDS(
  array(c(0.5, 0.3, 0.7, 0.9, 0.6, 0.7, 0.8), c(7, 5)),
  file.path(tempdir(), "hs_mean_1.RData")
)
# Generator
capacity_gen <- Generator$new(
  description = "capacity",
  region = region,
  time_steps = 5,
  spatial_correlation = spatial_correlation,
  temporal_correlation = 0.9,
  hs_sd = 0.1, # template attached
  inputs = c("hs_file", "density_max", "initial_n"),
  outputs = c("initial_abundance", "carrying_capacity")
)
capacity_gen$add_generative_requirements(list(
  hs_mean = "file",
  hs_sample = "distribution",
  carrying_capacity = "function",
  initial_abundance = "function"
))
# File template for mean habitat suitability
capacity_gen$add_file_template("hs_mean",
  path_template = file.path(tempdir(), "hs_mean_\%s.RData"),
  path_params = c("hs_file"), file_type = "RDS"
)
# Distribution template for sampling habitat suitability
capacity_gen$add_distribution_template("hs_sample",
  distr_type = "beta",
  distr_params = list(
    mean = "hs_mean",
    sd = "hs_sd"
  )
)
# Function templates for initial abundance and carrying capacity
capacity_gen$add_function_template("initial_abundance",
  function_def = function(params) {
    stats::rmultinom(1,
      size = params$initial_n,
      prob = params$hs_sample[, 1]
    )
  },
  call_params = c("initial_n", "hs_sample")
)
capacity_gen$add_function_template("carrying_capacity",
  function_def = function(params) {
    round(params$density_max * params$hs_sample)
  },
  call_params = c("density_max", "hs_sample")
)
# Generation
capacity_gen$generate(input_values = list(
  hs_file = 1,
  initial_n = 400,
  density_max = 100
))

}
\section{Super classes}{
\code{\link[poems:GenericClass]{poems::GenericClass}} -> \code{\link[poems:GenericModel]{poems::GenericModel}} -> \code{\link[poems:SpatialModel]{poems::SpatialModel}} -> \code{Generator}
}
\section{Public fields}{
\if{html}{\out{<div class="r6-fields">}}
\describe{
\item{\code{attached}}{A list of dynamically attached attributes (name-value pairs).}
}
\if{html}{\out{</div>}}
}
\section{Active bindings}{
\if{html}{\out{<div class="r6-active-bindings">}}
\describe{
\item{\code{model_attributes}}{A vector of model attribute names.}

\item{\code{region}}{A \code{\link{Region}} (or inherited class) object specifying the study region.}

\item{\code{coordinates}}{Data frame (or matrix) of X-Y population (WGS84) coordinates in longitude (degrees West) and latitude (degrees North) (get and set), or distance-based coordinates dynamically returned by region raster (get only).}

\item{\code{description}}{A brief description of what the generator generates.}

\item{\code{inputs}}{An array of input attribute names for the generator.}

\item{\code{outputs}}{An array of output attribute names for the generator.}

\item{\code{file_templates}}{A nested list of file template attributes.}

\item{\code{function_templates}}{A nested list of function template attributes.}

\item{\code{distribution_templates}}{A list of distribution template attributes.}

\item{\code{uses_correlations}}{A boolean to indicate that a \code{\link{SpatialCorrelation}} (or inherited class) object is used for generating correlated random deviates.}

\item{\code{spatial_correlation}}{A \code{\link{SpatialCorrelation}} (or inherited class) object for generating correlated random deviates.}

\item{\code{temporal_correlation}}{Absolute correlation coefficient between simulation time steps for all grid cells (0-1; default = 1).}

\item{\code{time_steps}}{Number of simulation time steps.}

\item{\code{generate_rasters}}{Boolean to indicate if rasters should be generated (defaults to TRUE when region uses rasters).}

\item{\code{decimals}}{Number of decimal places applied to generated data outputs (default: NULL = no rounding).}

\item{\code{occupancy_mask}}{Optional binary mask array (matrix), data frame, or raster (stack) for generated (time-series) data outputs.}

\item{\code{template_attached}}{A list of template-nested dynamically attached model attributes that are maintained via shallow or \emph{new} cloning.}

\item{\code{attribute_aliases}}{A list of alternative alias names for model attributes (form: \code{alias = "attribute"}) to be used with the set and get attributes methods.}

\item{\code{generative_template}}{A nested \code{\link{GenerativeTemplate}} (or inherited class) object for model attributes that are maintained via shallow or \emph{new} cloning.}

\item{\code{generative_requirements}}{A list of attribute names and the template setting (\emph{"file"}, \emph{"function"}, or \emph{"default"}) that is required to generate their values.}

\item{\code{error_messages}}{A vector of error messages encountered when setting model attributes.}

\item{\code{warning_messages}}{A vector of warning messages encountered when setting model attributes.}
}
\if{html}{\out{</div>}}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-Generator-new}{\code{Generator$new()}}
\item \href{#method-Generator-new_clone}{\code{Generator$new_clone()}}
\item \href{#method-Generator-get_attributes}{\code{Generator$get_attributes()}}
\item \href{#method-Generator-generate}{\code{Generator$generate()}}
\item \href{#method-Generator-add_file_template}{\code{Generator$add_file_template()}}
\item \href{#method-Generator-add_function_template}{\code{Generator$add_function_template()}}
\item \href{#method-Generator-add_distribution_template}{\code{Generator$add_distribution_template()}}
\item \href{#method-Generator-read_file}{\code{Generator$read_file()}}
\item \href{#method-Generator-run_function}{\code{Generator$run_function()}}
\item \href{#method-Generator-sample_distribution}{\code{Generator$sample_distribution()}}
\item \href{#method-Generator-add_generative_requirements}{\code{Generator$add_generative_requirements()}}
\item \href{#method-Generator-generative_requirements_satisfied}{\code{Generator$generative_requirements_satisfied()}}
\item \href{#method-Generator-clone}{\code{Generator$clone()}}
}
}
\if{html}{\out{
<details open><summary>Inherited methods</summary>
<ul>
<li><span class="pkg-link" data-pkg="poems" data-topic="GenericModel" data-id="get_attribute"><a href='../../poems/html/GenericModel.html#method-GenericModel-get_attribute'><code>poems::GenericModel$get_attribute()</code></a></span></li>
<li><span class="pkg-link" data-pkg="poems" data-topic="GenericModel" data-id="get_attribute_aliases"><a href='../../poems/html/GenericModel.html#method-GenericModel-get_attribute_aliases'><code>poems::GenericModel$get_attribute_aliases()</code></a></span></li>
<li><span class="pkg-link" data-pkg="poems" data-topic="GenericModel" data-id="get_attribute_names"><a href='../../poems/html/GenericModel.html#method-GenericModel-get_attribute_names'><code>poems::GenericModel$get_attribute_names()</code></a></span></li>
<li><span class="pkg-link" data-pkg="poems" data-topic="GenericModel" data-id="set_attributes"><a href='../../poems/html/GenericModel.html#method-GenericModel-set_attributes'><code>poems::GenericModel$set_attributes()</code></a></span></li>
</ul>
</details>
}}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Generator-new"></a>}}
\if{latex}{\out{\hypertarget{method-Generator-new}{}}}
\subsection{Method \code{new()}}{
Initialization method sets the generative template and requirements as well as any attributes passed via a \emph{params} list or individually.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Generator$new(generative_template = NULL, generative_requirements = NULL, ...)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{generative_template}}{A \code{\link{GenerativeTemplate}} (or inherited class) object containing the file, function and/or distribution templates utilized (facilitates shallow cloning).}

\item{\code{generative_requirements}}{A list of attribute names and the template setting ("file", "function", or "distribution") that is required to generate their values.}

\item{\code{...}}{Parameters passed via a \emph{params} list or individually.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Generator-new_clone"></a>}}
\if{latex}{\out{\hypertarget{method-Generator-new_clone}{}}}
\subsection{Method \code{new_clone()}}{
Creates a new (re-initialized) object of the current (inherited) object class with optionally passed parameters.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Generator$new_clone(...)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{...}}{Parameters passed via the inherited class constructor (defined in initialize and run via new).}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
New object of the current (inherited) class.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Generator-get_attributes"></a>}}
\if{latex}{\out{\hypertarget{method-Generator-get_attributes}{}}}
\subsection{Method \code{get_attributes()}}{
Returns a list of existing and template-generated values for selected attributes or attribute aliases (when array of parameter names provided), or all existing attributes (when no params).
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Generator$get_attributes(params = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{params}}{Array of attribute names to return, including those to be template-generated (all when NULL).}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
List of selected or all attributes values.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Generator-generate"></a>}}
\if{latex}{\out{\hypertarget{method-Generator-generate}{}}}
\subsection{Method \code{generate()}}{
Returns a list of generated output values (attributes) corresponding to the sample input values (attributes).
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Generator$generate(input_values = list())}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{input_values}}{List of sample input values for generator attributes.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
List containing generated model output attributes and/or any error/warning messages.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Generator-add_file_template"></a>}}
\if{latex}{\out{\hypertarget{method-Generator-add_file_template}{}}}
\subsection{Method \code{add_file_template()}}{
Adds a file template for reading raster/RData(RDS)/CSV files for a given
model attribute.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Generator$add_file_template(
  param,
  path_template,
  path_params = c(),
  file_type = "GRD"
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{param}}{Name of model attribute to be read from a file.}

\item{\code{path_template}}{Template string for the file path with placeholders
(see \code{\link{sprintf}}) for simulation sample parameters.}

\item{\code{path_params}}{Array of the names of the simulation sample
parameters to be substituted (in order) into the path template.}

\item{\code{file_type}}{File type raster \emph{"GRD"} (default), \emph{"TIF"},
\emph{"RData/RDS"}, \emph{"QS"}, or \emph{"CSV"} to be read.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Generator-add_function_template"></a>}}
\if{latex}{\out{\hypertarget{method-Generator-add_function_template}{}}}
\subsection{Method \code{add_function_template()}}{
Adds a function template for running a user-defined function to calculate a given model attribute.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Generator$add_function_template(param, function_def, call_params = c())}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{param}}{Name of model attribute to be generated using a function.}

\item{\code{function_def}}{Function definition (or path to the file containing the function) in form: \code{function(params)}, where \emph{params} is a list passed to the function.}

\item{\code{call_params}}{Array of the names of the model parameters/attributes to be passed into the function via a list: \emph{params}.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Generator-add_distribution_template"></a>}}
\if{latex}{\out{\hypertarget{method-Generator-add_distribution_template}{}}}
\subsection{Method \code{add_distribution_template()}}{
Adds a distribution template for generating a given model attribute via sampling a distribution.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Generator$add_distribution_template(
  param,
  distr_type = c("uniform", "normal", "lognormal", "beta", "triangular"),
  distr_params = list(),
  sample = NULL,
  random_seed = NULL,
  normalize_threshold = NULL
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{param}}{Name of model attribute to be generated via sampling a distribution.}

\item{\code{distr_type}}{Distribution type to sample from (uniform, normal, lognormal, beta or triangular).}

\item{\code{distr_params}}{List of distribution parameters and their values or associated model attributes (uniform: lower, upper; normal: mean, sd; lognormal: meanlog, sdlog (or mean, sd); beta: alpha, beta (or mean, sd); triangular: lower, mode, upper).}

\item{\code{sample}}{Model attribute(s) name(s) or values associated with single sample probabilities (0-1), or bounds as a vector (e.g. \code{sample = c("p_lower", "p_upper")}), or as a list (e.g. \code{sample = list(mid = "p", window = 0.2)} for bounds p +/- 0.1).}

\item{\code{random_seed}}{Random seed utilized when sample probability is generated internally, via bounds, and/or correlated deviates.}

\item{\code{normalize_threshold}}{Optional normalization threshold is utilized when generated values are to be normalized with a fixed upper limit/threshold.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Generator-read_file"></a>}}
\if{latex}{\out{\hypertarget{method-Generator-read_file}{}}}
\subsection{Method \code{read_file()}}{
Reads and returns the value of a model attribute from a file using the corresponding file template and simulation sample parameters.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Generator$read_file(param)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{param}}{Name of model attribute to be read from the file.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
Model attribute value read from a file.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Generator-run_function"></a>}}
\if{latex}{\out{\hypertarget{method-Generator-run_function}{}}}
\subsection{Method \code{run_function()}}{
Returns the calculated value of a model attribute using the corresponding function template and model simulation sample parameters.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Generator$run_function(param)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{param}}{Name of model attribute to be calculated using a function.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
Model attribute value calculated using a function.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Generator-sample_distribution"></a>}}
\if{latex}{\out{\hypertarget{method-Generator-sample_distribution}{}}}
\subsection{Method \code{sample_distribution()}}{
Returns the calculated value of a model attribute using the corresponding distribution template and simulation sample parameters.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Generator$sample_distribution(param)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{param}}{Name of model attribute to be calculated using a sampling distribution.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
Model attribute value calculated via distribution sampling.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Generator-add_generative_requirements"></a>}}
\if{latex}{\out{\hypertarget{method-Generator-add_generative_requirements}{}}}
\subsection{Method \code{add_generative_requirements()}}{
Adds attribute names and the template setting (\emph{"file"}, \emph{"function"} or \emph{"distribution"}) that is required to generate their values (via a \emph{params} list or individually).
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Generator$add_generative_requirements(params = list(), ...)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{params}}{Parameters passed via a list (e.g. \code{params = list(attr1 = "file", attr2 = "function", attr3 = "distribution")}).}

\item{\code{...}}{Parameters passed individually (e.g. \code{attr3 = "file"}).}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Generator-generative_requirements_satisfied"></a>}}
\if{latex}{\out{\hypertarget{method-Generator-generative_requirements_satisfied}{}}}
\subsection{Method \code{generative_requirements_satisfied()}}{
Returns a boolean to indicate that all the file, function and/or distribution template settings that are required for attribute generation are present.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Generator$generative_requirements_satisfied()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
Boolean to indicate that the required settings for attribute generation are present.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Generator-clone"></a>}}
\if{latex}{\out{\hypertarget{method-Generator-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Generator$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
