% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/measure_disease.R
\name{measure_disease}
\alias{measure_disease}
\alias{measure_disease_iter}
\title{Performs plant disease measurements}
\usage{
measure_disease(
  img,
  img_healthy = NULL,
  img_symptoms = NULL,
  img_background = NULL,
  pattern = NULL,
  parallel = FALSE,
  workers = NULL,
  resize = FALSE,
  fill_hull = TRUE,
  index_lb = NULL,
  index_dh = "GLI",
  threshold = NULL,
  invert = FALSE,
  lower_size = NULL,
  upper_size = NULL,
  topn_lower = NULL,
  topn_upper = NULL,
  randomize = TRUE,
  nsample = 3000,
  watershed = FALSE,
  lesion_size = "medium",
  tolerance = NULL,
  extension = NULL,
  show_features = FALSE,
  show_segmentation = FALSE,
  show_image = TRUE,
  show_original = TRUE,
  show_background = TRUE,
  show_contour = TRUE,
  contour_col = "white",
  contour_size = 1,
  col_leaf = NULL,
  col_lesions = NULL,
  col_background = NULL,
  marker = FALSE,
  marker_col = NULL,
  marker_size = NULL,
  save_image = FALSE,
  prefix = "proc_",
  dir_original = NULL,
  dir_processed = NULL,
  verbose = TRUE
)

measure_disease_iter(img, has_background = TRUE, r = 5, ...)
}
\arguments{
\item{img}{The image to be analyzed.}

\item{img_healthy}{A color palette of healthy areas.}

\item{img_symptoms}{A color palette of lesioned areas.}

\item{img_background}{An optional color palette of the image background.}

\item{pattern}{A pattern of file name used to identify images to be
processed. For example, if \code{pattern = "im"} all images that the name
matches the pattern (e.g., img1.-, image1.-, im2.-) will be analyzed.
Providing any number as pattern (e.g., \code{pattern = "1"}) will select
images that are named as 1.-, 2.-, and so on.}

\item{parallel}{Processes the images asynchronously (in parallel) in separate
R sessions running in the background on the same machine. It may speed up
the processing time, especially when \code{pattern} is used is informed. The
number of sections is set up to 70\% of available cores.}

\item{workers}{A positive numeric scalar or a function specifying the maximum
number of parallel processes that can be active at the same time.}

\item{resize}{Resize the image before processing? Defaults to \code{FALSE}. Use a
numeric value of range 0-100 (proportion of the size of the original
image).}

\item{fill_hull}{Fill holes in the image? Defaults to \code{TRUE}. This is useful
to fill holes in leaves, e.g., those caused by insect attack, ensuring the
hole area will be accounted for the leaf, not background.}

\item{index_lb}{The index used to segment the foreground (e.g., leaf) from
the background. If not declared, the entire image area (pixels) will be
considered in the computation of the severity.}

\item{index_dh}{The index used to segment diseased from healthy tissues when
\code{img_healthy} and \code{img_symptoms} are not declared. Defaults to \code{"GLI"}. See
\code{\link[=image_index]{image_index()}} for more details.}

\item{threshold}{By default (\code{threshold = NULL}), a threshold value based on
Otsu's method is used to reduce the grayscale image to a binary image. If a
numeric value is informed, this value will be used as a threshold. Inform
any non-numeric value different than "Otsu" to iteratively choose the
threshold based on a raster plot showing pixel intensity of the index. Must
be a vector of length 2 to indicate the threshold for \code{index_lb} and
\code{index_dh}, respectively.}

\item{invert}{Inverts the binary image if desired. This is useful to process
images with black background. Defaults to \code{FALSE}.}

\item{lower_size}{Lower limit for size for the image analysis. Leaf images
often contain dirt and dust. To prevent dust from affecting the image
analysis, the lower limit of analyzed size is set to 0.1, i.e., objects
with lesser than 10\% of the mean of all objects are removed. One can set a
known area or use \code{lower_limit = 0} to select all objects (not advised).}

\item{upper_size}{Upper limit for size for the image analysis. Defaults to
\code{NULL}, i.e., no upper limit used.}

\item{topn_lower, topn_upper}{Select the top \code{n} lesions based on its area.
\code{topn_lower} selects the \code{n} lesions with the smallest area whereas
\code{topn_upper} selects the \code{n} lesions with the largest area.}

\item{randomize}{Randomize the lines before training the model? Defaults to
\code{TRUE}.}

\item{nsample}{The number of sample pixels to be used in training step.
Defaults to \code{3000}.}

\item{watershed}{If \code{TRUE} (Default) implements the Watershed Algorithm to
segment lesions connected by a fairly few pixels that could be considered
as two distinct lesions. If \code{FALSE}, lesions that are connected by any
pixel are considered unique lesions. For more details see
\code{\link[EBImage:watershed]{EBImage::watershed()}}.}

\item{lesion_size}{The size of the lesion. Used to automatically tune
\code{tolerance} and \code{extension} parameters. One of the following. \code{"small"}
(2-5 mm in diameter, e.g, rust pustules), \code{"medium"} (0.5-1.0 cm in
diameter, e.g, wheat leaf spot), \code{"large"} (1-2 cm in diameter, and
\code{"elarge"} (2-3 cm in diameter, e.g, target spot of soybean).}

\item{tolerance}{The minimum height of the object in the units of image
intensity between its highest point (seed) and the point where it contacts
another object (checked for every contact pixel). If the height is smaller
than the tolerance, the object will be combined with one of its neighbors,
which is the highest. Defaults to \code{NULL}, i.e., starting values are set up
according to the argument \code{lesion_size}.}

\item{extension}{Radius of the neighborhood in pixels for the detection of
neighboring objects. Defaults to 20. Higher value smooths out small
objects.}

\item{show_features}{If \code{TRUE} returnS the lesion features such as number,
area, perimeter, and radius. Defaults to \code{FALSE}.}

\item{show_segmentation}{Shows the object segmentation colored with random
permutations. Defaults to \code{TRUE}.}

\item{show_image}{Show image after processing? Defaults to \code{TRUE}.}

\item{show_original}{Show the symptoms in the original image?}

\item{show_background}{Show the background? Defaults to \code{TRUE}. A white
background is shown by default when \code{show_original = FALSE}.}

\item{show_contour}{Show a contour line around the lesions? Defaults
to \code{TRUE}.}

\item{contour_col, contour_size}{The color and size for the contour line
around objects. Defaults to \code{contour_col = "white"} and \code{contour_size = 1}.}

\item{col_leaf}{Leaf color after image processing. Defaults to \code{"green"}}

\item{col_lesions}{Symptoms color after image processing. Defaults to
\code{"red"}.}

\item{col_background}{Background color after image processing. Defaults to
\code{"NULL"}.}

\item{marker, marker_col, marker_size}{The type, color and size of the object
marker. Defaults to \code{NULL}, which shows nothing. Use \code{marker = "point"} to
show a point in each lesion or \code{marker = "*"} where \code{"*"} is any variable
name of the \code{shape} data frame returned by the function.}

\item{save_image}{Save the image after processing? The image is saved in the
current working directory named as \verb{proc_*} where \code{*} is the image name
given in \code{img}.}

\item{prefix}{The prefix to be included in the processed images. Defaults to
\code{"proc_"}.}

\item{dir_original, dir_processed}{The directory containing the original and
processed images. Defaults to \code{NULL}. In this case, the function will
search for the image \code{img} in the current working directory. After
processing, when \code{save_image = TRUE}, the processed image will be also
saved in such a directory. It can be either a full path, e.g.,
\code{"C:/Desktop/imgs"}, or a subfolder within the current working directory,
e.g., \code{"/imgs"}.}

\item{verbose}{If \code{TRUE} (default) a summary is shown in the console.}

\item{has_background}{A logical indicating if the image has a background to
be segmented before processing.}

\item{r}{The radius of neighborhood pixels. Defaults to \code{5}. A square is
drawn indicating the selected pixels.}

\item{...}{Further parameters passed on to \code{measure_disease()}.}
}
\value{
\itemize{
\item \code{measure_disease()} returns a list with the following objects:
\itemize{
\item \code{severity} A data frame with the percentage of healthy and symptomatic
areas.
\item \code{shape},\code{statistics} If \code{show_features = TRUE} is used, returns the shape
(area, perimeter, etc.) for each lesion and a summary statistic of the
results.
}
\item \code{measure_disease_iter()} returns a list with the following objects:
\itemize{
\item \code{results} A list with the objects returned by \code{measure_disease()}.
\item \code{leaf} The color palettes for the healthy leaf.
\item \code{disease} The color palettes for the diseased leaf.
\item \code{background} The color palettes for the background.
}
}
}
\description{
\itemize{
\item \code{measure_disease()} computes the percentage of symptomatic leaf area and
(optionally) counts and compute shapes (area, perimeter, radius, etc.) of
lesions in a sample or entire leaf using color palettes. See more at
\strong{Details}.
\item \code{measure_disease_iter()} provides an iterative section for
\code{measure_disease()}, where the user picks up samples in the image to create
the needed color palettes.
}
}
\details{
In \code{measure_disease()}, a general linear model (binomial family) fitted to
the RGB values is used to segment the lesions from the healthy leaf. If a
pallet of background is provided, the function takes care of the details to
isolate it before computing the number and area of lesions. By using \code{pattern}
it is possible to process several images with common pattern names that are
stored in the current working directory or in the subdirectory informed in
\code{dir_original}.

If \code{img_healthy} and \code{img_symptoms} are not declared, RGB-based phenotyping of
foliar disease severity is performed using the index informed in \code{index_lb} to
first segment leaf from background and \code{index_dh} to segment diseased from
healthy tissues.

\code{measure_disease_iter()} only run in an interactive section. In this function,
users will be able to pick up samples of images to iteratively create the
needed color palettes. This process calls \code{\link[=pick_palette]{pick_palette()}} internally. If
\code{has_background} is TRUE (default) the color palette for the background is
first created. The sample of colors is performed in each left-button mouse
click and continues until the user press Esc. Then, a new sampling process is
performed to sample the color of healthy tissues and then diseased tissues.
The generated palettes are then passed on to measure_disease(). All the
arguments of such function can be passed using the ... (three dots).
}
\examples{
\donttest{
library(pliman)
img <- image_pliman("sev_leaf_nb.jpg")
healthy <- image_pliman("sev_healthy.jpg")
lesions <- image_pliman("sev_sympt.jpg")
image_combine(img, healthy, lesions, ncol = 3)

sev <-
 measure_disease(img = img,
                 img_healthy = healthy,
                 img_symptoms = lesions,
                 lesion_size = "large",
                 show_image = TRUE)

# an interactive section
measure_disease_iter(img)
}

}
\author{
Tiago Olivoto \email{tiagoolivoto@gmail.com}
}
