#' Build home section
#'
#' This function generates the home page, and converts other `.md` files
#' typically found in the package root and in `.github/`. It generates the home
#' page from  `index.md` or `README.md`, falling back to the description field
#' in `DESCRIPTION` if neither is found. It also builds an authors page
#' from the `DESCRIPTION` and `inst/CITATION` (if present) and
#' a license page, and converts any `.md` files found in
#'
#' @section Sidebar:
#' The sidebar is automatically populated with:
#'
#' *   Development status badges found in `README.md`/`index.md`. pkgdown
#'     identifies badge paragraphs in two ways:
#'
#'     * A paragraph starting with `<!-- badges: start -->` and ending with
#'       `<!-- badges: end -->` as created by `usethis::use_readme_md()`. or
#'       `usethis::use_readme_rmd()`.
#'
#'     * The first paragraph, if it only contains images.
#'
#' *   A link for bug reports is added if the `BugReports` field in
#'     `DESCRIPTION` contains a link. You can use `usethis::use_github_links()`
#'     to populate this field.
#'
#' *   Licensing information if `LICENSE`/`LICENCE` or `LICENSE.md`/`LICENCE.md`
#'     files are present.
#'
#' *   Community information is linked in the side bar using the
#'     `.github/CONTRIBUTING.md` and `.github/CODE_OF_CONDUCT.md` files,
#'     if present.
#'
#' *   Extra markdown files in the base directory or in `.github/` are copied
#'     to `docs/` and converted to HTML.
#'
#' *   Citation information from a `inst/CITATION` file is linked in the side bar
#'     to the [authors page](https://testthat.r-lib.org/authors.html).
#'
#' *   Author ORCID identification numbers in the `DESCRIPTION` are linked under
#'     "Developers" using the ORCID logo:
#'
#'     ```
#'     Authors@R: c(
#'         person("Hadley", "Wickham", , "hadley@rstudio.com", role = c("aut", "cre"),
#'           comment = c(ORCID = "0000-0003-4757-117X")
#'         ),
#'         person("Jay", "Hesselberth", role = "aut",
#'           comment = c(ORCID = "0000-0002-6299-179X")
#'         )
#'       )
#'     ```
#'
#' @section Images and figures:
#' If you want to include images in your `README.md`, they must be stored
#' somewhere in the package so that they can be displayed on the CRAN website.
#' The best place to put them is `man/figures`. If you are generating figures
#' with R Markdown, make sure you set up `fig.path` as followed:
#'
#' \preformatted{
#' ```\{r, include = FALSE\}
#' knitr::opts_chunk$set(
#'   fig.path = "man/figures/"
#' )
#' ```
#' }
#'
#' @section Package logo:
#' If you have a package logo, you can include it at the top of your README in a
#' level-one heading:
#'
#' ```
#' # pkgdown <img src="man/figures/logo.png" align="right" />
#' ```
#'
#' [init_site()] will also automatically create a favicon set from your package
#' logo.
#'
#' @section YAML config - home:
#' To tweak the home page, you need a section called `home`.
#'
#' By default, the page title and description are extracted automatically
#' from the `Title` and `Description` fields `DESCRIPTION` (stripping
#' single quotes off quoted words). CRAN ensures that these fields don't contain
#' phrases like "R package" because that's obvious on CRAN. To make your
#' package more findable with google, it's good practice to override the
#' `title` and `description`, thinking about what people might search for:
#'
#' ```
#' home:
#'   title: An R package for pool-noodle discovery
#'   description: >
#'     Do you love R? Do you love pool-noodles? If so, you might enjoy
#'     using this package to automatically discover and add pool-noodles
#'     to your growing collection.
#' ```
#'
#' (Note the use of YAML's `>`; this is a convenient way of writing paragraphs
#' of text.)
#'
#' The sidebar links are automatically generated by inspecting the `URL` and
#' `BugReports` fields of the `DESCRIPTION`. You can add additional links with a
#' subsection called `links`, which should contain a list of `text` + `href`
#' elements:
#'
#' ```
#' home:
#'   links:
#'   - text: Link text
#'     href: http://website.com
#' ```
#'
#' READMEs usually start with an `<h1>` containing the package name. If
#' that feels duplicative with the package name in the navbar you can
#' remove it with `strip_header: true`:
#'
#' ```
#' home:
#'   strip_header: true
#' ```
#'
#' @section YAML config - authors:
#' The "developers" list is populated by the maintainer ("cre"), authors
#' ("aut"), and funder ("fnd") from the `DESCRIPTION`. You can modify their
#' display on the home page by adding a subsection for `authors`. The author
#' name should match the value in `DESCRIPTION`:
#'
#' ```
#' authors:
#'   name:
#'     href: "http://name-website.com"
#'     html: "<img src='name-picture.png' height=24>"
#' ```
#' @inheritParams build_articles
#' @export
build_home <- function(pkg = ".",
                       override = list(),
                       preview = NA,
                       quiet = TRUE) {

  pkg <- section_init(pkg, depth = 0L, override = override)
  rule("Building home")
  dir_create(pkg$dst_path)

  if (has_citation(pkg$src_path)) {
    build_citation_authors(pkg)
  } else {
    build_authors(pkg)
  }
  build_home_md(pkg)
  build_home_license(pkg)
  build_home_index(pkg, quiet = quiet)

  preview_site(pkg, "/", preview = preview)
}
