% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/icc.R
\name{icc}
\alias{icc}
\alias{variance_decomposition}
\title{Intraclass Correlation Coefficient (ICC)}
\usage{
icc(
  model,
  by_group = FALSE,
  tolerance = 1e-05,
  ci = NULL,
  iterations = 100,
  ci_method = NULL,
  null_model = NULL,
  approximation = "lognormal",
  model_component = NULL,
  verbose = TRUE,
  ...
)

variance_decomposition(model, re_formula = NULL, robust = TRUE, ci = 0.95, ...)
}
\arguments{
\item{model}{A (Bayesian) mixed effects model.}

\item{by_group}{Logical, if \code{TRUE}, \code{icc()} returns the variance components
for each random-effects level (if there are multiple levels). See 'Details'.}

\item{tolerance}{Tolerance for singularity check of random effects, to decide
whether to compute random effect variances or not. Indicates up to which
value the convergence result is accepted. The larger tolerance is, the
stricter the test will be. See \code{\link[performance:check_singularity]{performance::check_singularity()}}.}

\item{ci}{Confidence resp. credible interval level. For \code{icc()}, \code{r2()}, and
\code{rmse()}, confidence intervals are based on bootstrapped samples from the
ICC, R2 or RMSE value. See \code{iterations}.}

\item{iterations}{Number of bootstrap-replicates when computing confidence
intervals for the ICC, R2, RMSE etc.}

\item{ci_method}{Character string, indicating the bootstrap-method. Should
be \code{NULL} (default), in which case \code{lme4::bootMer()} is used for bootstrapped
confidence intervals. However, if bootstrapped intervals cannot be calculated
this way, try \code{ci_method = "boot"}, which falls back to \code{boot::boot()}. This
may successfully return bootstrapped confidence intervals, but bootstrapped
samples may not be appropriate for the multilevel structure of the model.
There is also an option \code{ci_method = "analytical"}, which tries to calculate
analytical confidence assuming a chi-squared distribution. However, these
intervals are rather inaccurate and often too narrow. It is recommended to
calculate bootstrapped confidence intervals for mixed models.}

\item{null_model}{Optional, a null model to compute the random effect variances,
which is passed to \code{\link[insight:get_variance]{insight::get_variance()}}. Usually only required if
calculation of r-squared or ICC fails when \code{null_model} is not specified. If
calculating the null model takes longer and you already have fit the null
model, you can pass it here, too, to speed up the process.}

\item{approximation}{Character string, indicating the approximation method
for the distribution-specific (observation level, or residual) variance. Only
applies to non-Gaussian models. Can be \code{"lognormal"} (default), \code{"delta"} or
\code{"trigamma"}. For binomial models, the default is the \emph{theoretical}
distribution specific variance, however, it can also be
\code{"observation_level"}. See \emph{Nakagawa et al. 2017}, in particular supplement
2, for details.}

\item{model_component}{For models that can have a zero-inflation component,
specify for which component variances should be returned. If \code{NULL} or
\code{"full"} (the default), both the conditional and the zero-inflation component
are taken into account. If \code{"conditional"}, only the conditional component is
considered.}

\item{verbose}{Toggle warnings and messages.}

\item{...}{Arguments passed down to \code{lme4::bootMer()} or \code{boot::boot()}
for bootstrapped ICC, R2, RMSE etc.; for \code{variance_decomposition()},
arguments are passed down to \code{brms::posterior_predict()}.}

\item{re_formula}{Formula containing group-level effects to be considered in
the prediction. If \code{NULL} (default), include all group-level effects.
Else, for instance for nested models, name a specific group-level effect
to calculate the variance decomposition for this group-level. See 'Details'
and \code{?brms::posterior_predict}.}

\item{robust}{Logical, if \code{TRUE}, the median instead of mean is used to
calculate the central tendency of the variances.}
}
\value{
A list with two values, the adjusted ICC and the unadjusted ICC. For
\code{variance_decomposition()}, a list with two values, the decomposed
ICC as well as the credible intervals for this ICC.
}
\description{
This function calculates the intraclass-correlation coefficient (ICC) -
sometimes also called \emph{variance partition coefficient} (VPC) or
\emph{repeatability} - for mixed effects models. The ICC can be calculated for all
models supported by \code{\link[insight:get_variance]{insight::get_variance()}}. For models fitted with the
\strong{brms}-package, \code{icc()} might fail due to the large variety of
models and families supported by the \strong{brms}-package. In such cases, an
alternative to the ICC is the \code{variance_decomposition()}, which is based
on the posterior predictive distribution (see 'Details').
}
\details{
\subsection{Interpretation}{

The ICC can be interpreted as "the proportion of the variance explained by
the grouping structure in the population". The grouping structure entails
that measurements are organized into groups (e.g., test scores in a school
can be grouped by classroom if there are multiple classrooms and each
classroom was administered the same test) and ICC indexes how strongly
measurements in the same group resemble each other. This index goes from 0,
if the grouping conveys no information, to 1, if all observations in a group
are identical (\emph{Gelman and Hill, 2007, p. 258}). In other word, the ICC -
sometimes conceptualized as the measurement repeatability - "can also be
interpreted as the expected correlation between two randomly drawn units
that are in the same group" \emph{(Hox 2010: 15)}, although this definition might
not apply to mixed models with more complex random effects structures. The
ICC can help determine whether a mixed model is even necessary: an ICC of
zero (or very close to zero) means the observations within clusters are no
more similar than observations from different clusters, and setting it as a
random factor might not be necessary.
}

\subsection{Difference with R2}{

The coefficient of determination R2 (that can be computed with \code{\link[=r2]{r2()}})
quantifies the proportion of variance explained by a statistical model, but
its definition in mixed model is complex (hence, different methods to compute
a proxy exist). ICC is related to R2 because they are both ratios of
variance components. More precisely, R2 is the proportion of the explained
variance (of the full model), while the ICC is the proportion of explained
variance that can be attributed to the random effects. In simple cases, the
ICC corresponds to the difference between the \emph{conditional R2} and the
\emph{marginal R2} (see \code{\link[=r2_nakagawa]{r2_nakagawa()}}).
}

\subsection{Calculation}{

The ICC is calculated by dividing the random effect variance,
\ifelse{html}{\out{&sigma;<sup>2</sup><sub>i</sub>}}{\eqn{\sigma^2_i}}, by
the total variance, i.e. the sum of the random effect variance and the
residual variance, \ifelse{html}{\out{&sigma;<sup>2</sup><sub>&epsilon;</sub>}}{\eqn{\sigma^2_\epsilon}}.
}

\subsection{Adjusted and unadjusted ICC}{

\code{icc()} calculates an adjusted and an unadjusted ICC, which both take all
sources of uncertainty (i.e. of \emph{all random effects}) into account. While
the \emph{adjusted ICC} only relates to the random effects, the \emph{unadjusted ICC}
also takes the fixed effects variances into account, more precisely, the
fixed effects variance is added to the denominator of the formula to
calculate the ICC (see \emph{Nakagawa et al. 2017}). Typically, the \emph{adjusted}
ICC is of interest when the analysis of random effects is of interest.
\code{icc()} returns a meaningful ICC also for more complex random effects
structures, like models with random slopes or nested design (more than two
levels) and is applicable for models with other distributions than Gaussian.
For more details on the computation of the variances, see
\code{?insight::get_variance}.
}

\subsection{ICC for unconditional and conditional models}{

Usually, the ICC is calculated for the null model ("unconditional model").
However, according to \emph{Raudenbush and Bryk (2002)} or
\emph{Rabe-Hesketh and Skrondal (2012)} it is also feasible to compute the
ICC for full models with covariates ("conditional models") and compare how
much, e.g., a level-2 variable explains the portion of variation in the
grouping structure (random intercept).
}

\subsection{ICC for specific group-levels}{

The proportion of variance for specific levels related to the overall model
can be computed by setting \code{by_group = TRUE}. The reported ICC is
the variance for each (random effect) group compared to the total
variance of the model. For mixed models with a simple random intercept,
this is identical to the classical (adjusted) ICC.
}

\subsection{Variance decomposition for brms-models}{

If \code{model} is of class \code{brmsfit}, \code{icc()} might fail due to the large
variety of models and families supported by the \strong{brms} package. In such
cases, \code{variance_decomposition()} is an alternative ICC measure. The function
calculates a variance decomposition based on the posterior predictive
distribution. In this case, first, the draws from the posterior predictive
distribution \emph{not conditioned} on group-level terms
(\code{posterior_predict(..., re_formula = NA)}) are calculated as well as draws
from this distribution \emph{conditioned} on \emph{all random effects} (by default,
unless specified else in \code{re_formula}) are taken. Then, second, the variances
for each of these draws are calculated. The "ICC" is then the ratio between
these two variances. This is the recommended way to analyse
random-effect-variances for non-Gaussian models. It is then possible to
compare variances across models, also by specifying different group-level
terms via the \code{re_formula}-argument.

Sometimes, when the variance of the posterior predictive distribution is
very large, the variance ratio in the output makes no sense, e.g. because
it is negative. In such cases, it might help to use \code{robust = TRUE}.
}
}
\section{Supported models and model families}{

The single variance components that are required to calculate the marginal
and conditional r-squared values are calculated using the \code{\link[insight:get_variance]{insight::get_variance()}}
function. The results are validated against the solutions provided by
\emph{Nakagawa et al. (2017)}, in particular examples shown in the Supplement 2
of the paper. Other model families are validated against results from the
\strong{MuMIn} package. This means that the r-squared values returned by \code{r2_nakagawa()}
should be accurate and reliable for following mixed models or model families:
\itemize{
\item Bernoulli (logistic) regression
\item Binomial regression (with other than binary outcomes)
\item Poisson and Quasi-Poisson regression
\item Negative binomial regression (including nbinom1 and nbinom2 families)
\item Gaussian regression (linear models)
\item Gamma regression
\item Tweedie regression
\item Beta regression
\item Ordered beta regression
}

Following model families are not yet validated, but should work:
\itemize{
\item Zero-inflated and hurdle models
\item Beta-binomial regression
\item Compound Poisson regression
\item Generalized Poisson regression
\item Log-normal regression
}

Extracting variance components for models with zero-inflation part is not
straightforward, because it is not definitely clear how the distribution-specific
variance should be calculated. Therefore, it is recommended to carefully
inspect the results, and probably validate against other models, e.g. Bayesian
models (although results may be only roughly comparable).

Log-normal regressions (e.g. \code{lognormal()} family in \strong{glmmTMB} or \code{gaussian("log")})
often have a very low fixed effects variance (if they were calculated as
suggested by \emph{Nakagawa et al. 2017}). This results in very low ICC or
r-squared values, which may not be meaningful.
}

\examples{
\dontshow{if (require("lme4")) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
model <- lme4::lmer(Sepal.Length ~ Petal.Length + (1 | Species), data = iris)
icc(model)

# ICC for specific group-levels
data(sleepstudy, package = "lme4")
set.seed(12345)
sleepstudy$grp <- sample(1:5, size = 180, replace = TRUE)
sleepstudy$subgrp <- NA
for (i in 1:5) {
  filter_group <- sleepstudy$grp == i
  sleepstudy$subgrp[filter_group] <-
    sample(1:30, size = sum(filter_group), replace = TRUE)
}
model <- lme4::lmer(
  Reaction ~ Days + (1 | grp / subgrp) + (1 | Subject),
  data = sleepstudy
)
icc(model, by_group = TRUE)
\dontshow{\}) # examplesIf}
}
\references{
\itemize{
\item Hox, J. J. (2010). Multilevel analysis: techniques and applications
(2nd ed). New York: Routledge.
\item Nakagawa, S., Johnson, P. C. D., and Schielzeth, H. (2017). The
coefficient of determination R2 and intra-class correlation coefficient
from generalized linear mixed-effects models revisited and expanded.
Journal of The Royal Society Interface, 14(134), 20170213.
\item Rabe-Hesketh, S., and Skrondal, A. (2012). Multilevel and longitudinal
modeling using Stata (3rd ed). College Station, Tex: Stata Press
Publication.
\item Raudenbush, S. W., and Bryk, A. S. (2002). Hierarchical linear models:
applications and data analysis methods (2nd ed). Thousand Oaks: Sage
Publications.
}
}
