% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/summ_hdr.R
\name{summ_hdr}
\alias{summ_hdr}
\title{Summarize distribution with Highest Density Region}
\usage{
summ_hdr(f, level = 0.95)
}
\arguments{
\item{f}{A pdqr-function representing distribution.}

\item{level}{A desired lower bound for a total probability of an output set
of intervals.}
}
\value{
A data frame with one row representing one closed interval of HDR and
the following columns:
\itemize{
\item \strong{left} \verb{<dbl>} : Left end of intervals.
\item \strong{right} \verb{<dbl>} : Right end of intervals.
}
}
\description{
\code{summ_hdr()} computes a Highest Density Region (HDR) of some pdqr-function
for a supplied \code{level}: a union of (closed) intervals total probability of
which is not less than \code{level} and probability/density at any point inside it
is bigger than some threshold (which should be maximum one with a property
of HDR having total probability not less than \code{level}). This also represents
a set of intervals with the lowest total width among all sets with total
probability not less than a \code{level}.
}
\details{
General algorithm of \code{summ_hdr()} consists from two steps:
\enumerate{
\item \strong{Find "target height"}. That is a value of probability/density which
divides all \link[=meta_support]{support} into two sets: the one with
probability/density not less than target height (it is a desired HDR) and the
other - with strictly less. The first set should also have total probability
not less than \code{level}.
\item \strong{Form a HDR as a set of closed intervals}.
}

If \code{f} has "discrete" type, target height is computed by looking at "x"
values of \link[=meta_x_tbl]{"x_tbl" metadata} in order of decreasing probability
until their total probability is not less than \code{level}. After that, all "x"
values with probability not less than height are considered to form a HDR.
Output is formed as a set of \strong{closed} intervals (i.e. both edges included)
inside of which lie all HDR "x" elements and others - don't.

If \code{f} has "continuous" type, target height is estimated as \code{1-level}
quantile of \code{Y = d_f(X)} distribution, where \code{d_f} is d-function
corresponding to \code{f} (\code{\link[=as_d]{as_d(f)}} in other words) and \code{X} is a random
variable represented by \code{f}. Essentially, \code{Y} has a distribution of \code{f}'s
density values and its \code{1-level} quantile is a target height. After that, HDR
is formed as a set of intervals \strong{with positive width} (if \code{level} is more
than 0, see Notes) inside which density is not less than target height.

\strong{Notes}:
\itemize{
\item If \code{level} is 0, output has one interval of zero width at point of \link[=summ_mode]{global mode}.
\item If \code{level} is 1, output has one interval equal to support.
\item Computation of target height in case of "continuous" type is approximate
which in some extreme cases (for example, like \link[=form_tails]{winsorized}
distributions) can lead to HDR having total probability very approximate to
and even slightly lower than \code{level}.
\item If d-function has "plateaus" (consecutive values with equal
probability/density) at computed target height, total probability of HDR can
be considerably bigger than \code{level} (see examples). However, this aligns with
HDR definition, as density values should be \strong{not less} than target height
and total probability should be \strong{not less} than \code{level}.
}
}
\examples{
# "discrete" functions
d_dis <- new_d(data.frame(x = 1:4, prob = c(0.4, 0.2, 0.3, 0.1)), "discrete")
summ_hdr(d_dis, 0.3)
summ_hdr(d_dis, 0.5)
summ_hdr(d_dis, 0.9)
## Zero width interval at global mode
summ_hdr(d_dis, 0)

# "continuous" functions
d_norm <- as_d(dnorm)
summ_hdr(d_norm, 0.95)
## Zero width interval at global mode
summ_hdr(d_norm, 0)

# Works well with mixture distributions
d_mix <- form_mix(list(as_d(dnorm), as_d(dnorm, mean = 5)))
summ_hdr(d_mix, 0.95)

# Plateaus
d_unif <- as_d(dunif)
## Returns all support because of density "plateau"
summ_hdr(d_unif, 0.1)

# Draw HDR
plot(d_mix)
region_draw(summ_hdr(d_mix, 0.95))
}
\seealso{
\code{\link[=region]{region_*()}} family of functions for working with output
HDR.

\code{\link[=summ_interval]{summ_interval()}} for computing of single interval summary of distribution.

Other summary functions: 
\code{\link{summ_center}()},
\code{\link{summ_classmetric}()},
\code{\link{summ_distance}()},
\code{\link{summ_entropy}()},
\code{\link{summ_interval}()},
\code{\link{summ_moment}()},
\code{\link{summ_order}()},
\code{\link{summ_prob_true}()},
\code{\link{summ_pval}()},
\code{\link{summ_quantile}()},
\code{\link{summ_roc}()},
\code{\link{summ_separation}()},
\code{\link{summ_spread}()}
}
\concept{summary functions}
