% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lod.R
\name{lod}
\alias{lod}
\title{Two-point LOD score}
\usage{
lod(
  x,
  markers = seq_len(x$nMark),
  theta = 0,
  loop_breakers = NULL,
  max.only = FALSE,
  verbose = FALSE,
  tol = 0.01
)
}
\arguments{
\item{x}{a \code{\link{linkdat}} object.}

\item{markers}{an integer vector denoting which markers to use.}

\item{theta}{either a numeric containing specific recombination ratio(s), or
the word 'max', indicating that the recombination ratio should be optimized
by the program.}

\item{loop_breakers}{a numeric containing IDs of individuals to be used as
loop breakers. Relevant only if the pedigree has loops. See
\code{\link{breakLoops}}.}

\item{max.only}{a logical indicating whether only the maximum LOD score
should be returned.}

\item{verbose}{a logical: verbose output or not.}

\item{tol}{a numeric passed on to \code{\link{optimize}} as its tolerance
parameter.}
}
\value{
If \code{max.only=TRUE}, the highest computed LOD score is returned,
  as a single number.

  Otherwise a \code{linkres} object, which is essentially a matrix containing
  the LOD scores. The details depend on the other parameters:

  If \code{theta} is numeric, the matrix has dimensions \code{length(theta) *
  length(markers)}, and the entry in row \code{t}, column \code{m} is the lod
  score of the pedigree for marker \code{m} assuming a recombination rate of
  \code{t}.

  If \code{theta='max'}, the \code{linkres} matrix has one column per marker
  and two rows: The first containing the LOD score and the second the optimal
  recombination ratio for each marker.

  If a marker has incompatible values (i.e. if a child of homozygous 1/1
  parents has a 2 allele), the corresponding output entries are \code{NaN}.
}
\description{
Calculates the two-point LOD scores of a pedigree for the specified markers.
The recombination ratio between the disease and marker loci can be either
fixed at specific values, or optimized.
}
\details{
The LOD score of a marker is defined as \deqn{LOD(\theta) =
\log[10]\frac{L(\theta)}{L(0.5)}} where \eqn{L(\theta)} denotes the
likelihood of the observed marker genotypes given a recombination ratio
\eqn{\theta} between the marker and the disease locus.
}
\examples{

x = linkdat(toyped, model=1)
res = lod(x)
res_theta = lod(x, theta=c(0, 0.1, 0.2, 0.5))
res_max = lod(x, theta='max')
stopifnot(all(0.3 == round(c(res, res_theta['0',], res_max['LOD',]), 1)))

# bigger pedigree with several markers
y = linkdat(dominant)
y = setModel(y, model=1, penetrances=c(.001, .9, .99))
lod(y, markers=305:310)
lod(y, markers=305:310, theta='max')

# Example with pedigree with loops:
z = linkdat(twoloops, model=2) # fully penetrant autosomal recessive model.

# add SNP for which individuals 15 and 16 are homozygous for the rare allele.
m = marker(z, 15:16, c(1,1), alleles=1:2, afreq=c(0.001, 0.999))
z = addMarker(z, m)
res1 = lod(z)
# manual specification of loop breakers gives same result
res2 = lod(z, loop_breakers=c(8,12))

# making the marker triallelic and adding some genotypes.
z = modifyMarker(z, marker=1, ids=c(7,9,11,13), genotype=3, alleles=1:3, afreq=c(0.001, 0.499, 0.5))
plot(z, 1)
res3 = lod(z)

z = modifyMarker(z, marker=1, alleles=1:4, afreq=c(0.001, 0.499, 0.25, 0.25))
res4 = lod(z)

stopifnot(all(3 == round(c(res1, res2, res3, res4), 1)))

}
\seealso{
\code{\link{likelihood}}, \code{\link{optimize}},
  \code{\link{breakLoops}}
}
