\name{ccPower}
\Rdversion{1.4}
\alias{ccPower}
\title{
	Simulation-based estimation of power for the case-control design
}
\description{
  Monte Carlo based estimation of statistical power for maximum likelihood estimator (MLE) of the components of a logistic regression model, based on the case-control design.
}
\usage{
ccPower(B=1000, betaTruth, X, N, expandX="all", etaTerms=NULL,
        nCC, r=1, alpha=0.05,
        digits=1, betaNames=NULL, monitor=NULL)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{B}{
   The number of datasets generated by the simulation.
}
  \item{betaTruth}{
    Regression coefficients from the logistic regression model.
}
  \item{X}{
    Design matrix for the logistic regression model. The first column should correspond to intercept. For each exposure, the baseline group should be coded as 0, the first level as 1, and so on.
}
  \item{N}{
    A numeric vector providing the sample size for each row of the design matrix, \code{X}. 
}
  \item{expandX}{
    Character vector indicating which columns of \code{X} to expand as a series of dummy variables. Useful when at least one exposure is continuous (and should not be expanded). Default is `all'; other options include `none' or character vector of column names. See Details, below.
  }
  \item{etaTerms}{
    Character vector indicating which columns of \code{X} are to be included in the model. See Details, below.
  }
  \item{nCC}{
    A numeric value indicating the total case-control sample size. If a vector is provided, separate simulations are run for each value.
}
  \item{r}{
    A numeric value indicating the control:case ratio in the case-control sample.
}
  \item{alpha}{
    Type I error rate assumed for the evaluation of coverage probabilities and power.
}
  \item{digits}{
    Integer indicating the precision to be used for the output.
}
  \item{betaNames}{
    An optional character vector of names for the regression coefficients, \code{betaTruth}.
}
\item{monitor}{
	Numeric value indicating how often \code{ccPower()} reports real-time progress on the simulation, as the \code{B} datasets are generated and evaluated. The default of \code{NULL} indicates no output.
      }     
}
\details{
	A simulation study is conducted to evaluate statistical power for the MLE of a logistic regression model, based on the case-control design. The overall simulation approach is the same as that described in \code{\link{ccSim}}. Power is estimated as the proportion of simulated datasets for which a hypothesis test of no effect is rejected. Each hypothesis test is performed using the generic \code{\link{glm}} function.
	
  The correspondence between \code{betaTruth} and \code{X}, specifically the ordering of elements, is based on successive use of \code{\link{factor}} to each column of \code{X} which is expanded via the \code{expandX} argument. Each exposure that is expanded must conform to a {0, 1, 2, ...} integer-based coding convention.
  
  The \code{etaTerms} argument is useful when only certain columns in \code{X} are to be included in the model.
  	
	A balanced case-control design is specified by setting \code{r}=1; setting \code{r}=2 indicates twice as many controls are sampled, relative to the number cases, from the total \code{nCC}.
	
  When evaluating operating characteristics of the MLE, some simulated datasets may result in unusually large or small estimates. Particularly, when the the case-control sample size, \code{nCC}, is small. In some settings, it may be desirable to truncate the Monte Carlo sampling distribution prior to evaluating operating characteristics. The \code{threshold} argument indicates the interval beyond which MLEs are ignored. The default is such that all \code{B} datasets are kept.
}
\value{
  \code{ccPower()} returns an object of class "ccPower", a list containing all the input arguments, as well as the following components:
    \item{betaPower}{
    	Power against the null hypothesis that the regression coefficient is zero for a Wald-based test with an \code{alpha} type I error rate.
    }
    \item{failed}{
    	A vector consisting of the number of datasets excluded from the power calculations (i.e. set to \code{NA}), for each simulation performed. For power calculations, the two reasons are: (1) lack of convergence indicated by \code{NA} point estimates returned by \code{\link{glm}}, (2) lack of convergence indicated by \code{NA} standard error point estimates returned by \code{\link{glm}}.
    }
}
\note{
	A generic print method provides formatted output of the results.

	A generic plot function \code{\link{plotPower}} provides plots of powers against different sample sizes for each estimate of a regression coefficient.
}
\references{
  Prentice, R. and Pyke, R. (1979) "Logistic disease incidence models and case-control studies." Biometrika 66:403-411.
}
\author{
  Sebastien Haneuse, Takumi Saegusa
}
\examples{
##
data(Ohio)

## 
XM   <- cbind(Int=1, Ohio[,1:3])
fitM <- glm(cbind(Death, N-Death) ~ factor(Age) + Sex + Race, data=Ohio,
            family=binomial)
betaNamesM <- c("Int", "Age1", "Age2", "Sex", "Race")

## Power for a single CC design
##
\dontrun{
ccResult1 <- ccPower(B=1000, betaTruth=fitM$coef, X=XM, N=Ohio$N, r=1,
                     nCC=500, betaNames=betaNamesM)
ccResult1}

## Power for the CC design, based on a balanced design with 
## various sample sizes
##
\dontrun{
ccResult2 <- ccPower(B=1000, betaTruth=fitM$coef, X=XM, N=Ohio$N, r=1,
                     nCC=seq(from=100, to=500, by=50), betaNames=betaNamesM)
ccResult2}

## Recalculate power for the setting where the age coefficients are
## halved from their observed true values
##   * the intercept is modified, accordingly, using the beta0() function
##
newBetaM      <- fitM$coef
newBetaM[2:3] <- newBetaM[2:3] / 2
newBetaM[1]   <- beta0(betaX=newBetaM[-1], X=XM, N=Ohio$N,
rhoY=sum(Ohio$Death)/sum(Ohio$N))
##
\dontrun{
ccResult3 <- ccPower(B=1000, betaTruth=newBetaM, X=XM, N=Ohio$N,
                     r=1, nCC=seq(from=100, to=500, by=50),
                     betaNames=betaNamesM)
ccResult3}
}

