% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/preprocessing.R
\name{preprocess_modality_t1}
\alias{preprocess_modality_t1}
\title{Preprocess T1-weighted MRI scan for one patient}
\usage{
preprocess_modality_t1(
  mri.patient,
  folder.patient,
  atlas,
  mask,
  inhomogeneity = "N4",
  transformation = "SyN"
)
}
\arguments{
\item{mri.patient}{path of the T1-weighted scan.}

\item{folder.patient}{folder containing the T1-weighted scan. This folder usually refers to the patient.}

\item{atlas}{atlas template in NifTI format to spatially register the T1-weighted scans. By default the MNI152 atlas template is used.}

\item{mask}{brain mask in NifTI format of the atlas template to performed the skull stripping.}

\item{inhomogeneity}{inhomogeneity correction algorithm to be applied. The correction by default is the 'N4' bias correction.}

\item{transformation}{non-linear transformation for registering the T1-w MRI scan to the reference template. 'SyN' transformation is used by default.}
}
\value{
paths of preprocessed MRI scans.
}
\description{
This function preprocesses a raw T1-w MRI scan and
generates a segmentation MRI scan using the FAST algorithm.
The preprocesising steps comprises imhomogeneity correction
'N4', registration to the MNI152 template with isotropic voxel size of 1mm^3
using the 'SyN' transformation, and skull stripping.
}
\examples{

\dontrun{
# Get general folder
folder <- system.file("extdata", package = "neurodata")
# Get covariates
covariates <- system.file("covariates.txt", package = "neurodata")
# Read covariates information
clinical_info <- read.csv(file = covariates, sep = ';')

# Folder and T1-weighted file of the patient
patient_folder <- file.path(folder,"patient01")
patient_T1 <- file.path(patient_folder,"patient01_T1.nii.gz")

# Getting preferred atlas template and template mask
# Using the MNI152 template available in the MNITemplate package
library(MNITemplate)
atlas <- getMNIPath()
atlas_mask <- readMNI("Brain_Mask")

# Preprocessing the patient's sequences
patient_preprocessed_mri <- preprocess_modality_t1(mri.patient = patient_T1,
                                                  folder.patient = patient_folder,
                                                  atlas = atlas, mask = atlas_mask,
                                                  inhomogeneity = 'N4',
                                                  transformation = 'SyN')


}

}
\references{
Nicholas J. Tustison, Brian B. Avants, Philip A. Cook, Yuanjie Zheng, Alexander Egan, Paul A. Yushkevich, and James C. Gee. N4ITK: Improved N3 Bias Correction. IEEE Trans Med Imaging, 29:1310–1320, 2010.

B. B. Avants, C. L. Epstein, M Grossman, J. C. Gee Symmetric diffeomorphic image registration with cross-correlation: evaluating automated labeling of elderly and neurodegenerative brain. Medical Image Analysis, 12:1310–1320, 2008.

Evans, A.C., Fox, P.T., Lancaster, J., Zilles, K., Woods, R., Paus, T., Simpson, G., Pike, B., Holmes, C., Collins, D.L., Thompson, P., MacDonald, D., Iacoboni, et al. A probabilistic atlas and reference system for the human brain: International Consortium for Brain Mapping (ICBM). Philos. Trans. R. Soc. London B Biol, 356:1293-1322, 2001.

Yongyue Zhang, J. Michael Brady, Stephen Smith Hidden Markov random field model for segmentation of brain MR image. Medical Imaging 2000: Image Processing, 2000.

Jean-Philippe Fortin, Elizabeth M Sweeney, John Muschelli, Ciprian M Crainiceanu, Russell T Shinohara, Alzheimer’s Disease Neuroimaging Initiative, et al. Removing inter-subject technical variability in magnetic resonance imaging studies. NeuroImage, 132:198–212, 2016.
}
\author{
David Payares
}
