% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/integration.R
\name{add_integration}
\alias{add_integration}
\alias{add_integration.default}
\alias{add_integration.data.frame}
\alias{add_integration.nma_data}
\alias{unnest_integration}
\title{Add numerical integration points to aggregate data}
\usage{
add_integration(x, ...)

\method{add_integration}{default}(x, ...)

\method{add_integration}{data.frame}(
  x,
  ...,
  cor = NULL,
  cor_adjust = NULL,
  n_int = 64L,
  int_args = list()
)

\method{add_integration}{nma_data}(
  x,
  ...,
  cor = NULL,
  cor_adjust = NULL,
  n_int = 64L,
  int_args = list()
)

unnest_integration(data)
}
\arguments{
\item{x}{An \code{nma_data} object, as created by the \verb{set_*()} functions or
\code{combine_network()}, or data frame}

\item{...}{Distributions for covariates, see "Details"}

\item{cor}{Correlation matrix to use for generating the integration points.
By default, this takes a weighted correlation matrix from all IPD studies.
Rows and columns should match the order of covariates specified in \code{...}.}

\item{cor_adjust}{Adjustment to apply to the correlation matrix given by
\code{cor} (or computed from the IPD if \code{cor = NULL}) to obtain the Gaussian
copula correlations, either \code{"spearman"}, \code{"pearson"}, or \code{"none"}, see
"Details". The default when \code{cor = NULL} is \code{"spearman"}, otherwise the
default is \code{"pearson"}.}

\item{n_int}{Number of integration points to generate, default 64. Powers of
2 are recommended, as these are expected to be particularly efficient for
QMC integration.}

\item{int_args}{A named list of arguments to pass to
\code{\link[randtoolbox:quasiRNG]{sobol()}}}

\item{data}{Data frame with nested integration points, stored in list
columns as \verb{.int_<variable name>}}
}
\value{
For the \code{nma_data} method, an object of class \link{nma_data}. For the
\code{data.frame} method, the input data frame is returned (as a \link{tibble}) with
an added column for each covariate (prefixed with ".int_"), containing the
numerical integration points nested as length-\code{n_int} vectors within each
row. For \code{unnest_integration()}, a data frame with integration points
unnested.
}
\description{
The \code{add_integration()} generic creates Quasi-Monte Carlo numerical
integration points using a Gaussian copula and Sobol' sequences, as described
in \insertCite{methods_paper;textual}{multinma}. Methods are available for
networks stored in \code{nma_data} objects, and for data frames. The function
\code{unnest_integration()} unnests integration points stored in a data frame, to
aid plotting or other exploration.
}
\details{
The arguments passed to \code{...} specify distributions for the
covariates. Argument names specify the name of the covariate, which should
match a covariate name in the IPD (if IPD are present). The required
marginal distribution is then specified using the function \code{\link[=distr]{distr()}}.

The argument \code{cor_adjust} specifies how the correlation matrix given by
\code{cor} (or computed from the IPD if \code{cor = NULL}) is adjusted to obtain the
correlation matrix for the Gaussian copula, using the formulae in
\insertCite{Xiao2018;textual}{multinma}.
\itemize{
\item \code{cor_adjust = "spearman"} should be used when the correlations \code{cor} have
been computed using Spearman's rank correlation. Correlations between
continuous covariates will be reproduced exactly by the integration points.
Correlations between discrete covariates will be reproduced approximately.
This is the default when \code{cor = NULL} and correlations are calculated from
the IPD studies.
\item \code{cor_adjust = "pearson"} should be used when the correlations \code{cor} have
been computed using Pearson's product-moment correlation. Correlations between
Normal covariates will be reproduced exactly by the integration points, all
others will be reproduced approximately. Correlations between discrete
covariates will be reproduced approximately (and identically to \code{cor_adjust   = "spearman"}). This is the default when \code{cor} is provided by the user,
since \code{\link[=cor]{cor()}} defaults to \code{method = "pearson"} and Pearson correlations are
most likely reported in published data. However, we recommend providing
Spearman correlations (e.g. from \code{cor(., method = "spearman")}) and using
\code{cor_adjust = "spearman"} where possible.
\item \code{cor_adjust = "none"} allows the user to specify the correlation matrix
for the Gaussian copula directly; no adjustment is applied.
\item \code{cor_adjust = "legacy"} is also available, which reproduces exactly the
behaviour from version 0.3.0 and earlier. This is similar to \code{cor_adjust =   "none"}, but unadjusted Spearman correlations are used if \code{cor = NULL}.
}

When adding integration points to a network object the correlation matrix
used is stored in \verb{$int_cor}, and the copula correlation matrix and
adjustment used are stored as attributes of \verb{$int_cor}. If this correlation
matrix is passed again to \code{add_integration()} (e.g. to reuse the
correlations for an external target population) this will be detected, and
the correct setting for \code{cor_adjust} will automatically be applied.
}
\examples{
## Plaque psoriasis ML-NMR - network setup and adding integration points
# Set up plaque psoriasis network combining IPD and AgD
library(dplyr)
pso_ipd <- filter(plaque_psoriasis_ipd,
                  studyc \%in\% c("UNCOVER-1", "UNCOVER-2", "UNCOVER-3"))

pso_agd <- filter(plaque_psoriasis_agd,
                  studyc == "FIXTURE")

head(pso_ipd)
head(pso_agd)

pso_ipd <- pso_ipd \%>\%
  mutate(# Variable transformations
    bsa = bsa / 100,
    prevsys = as.numeric(prevsys),
    psa = as.numeric(psa),
    weight = weight / 10,
    durnpso = durnpso / 10,
    # Treatment classes
    trtclass = case_when(trtn == 1 ~ "Placebo",
                         trtn \%in\% c(2, 3, 5, 6) ~ "IL blocker",
                         trtn == 4 ~ "TNFa blocker"),
    # Check complete cases for covariates of interest
    complete = complete.cases(durnpso, prevsys, bsa, weight, psa)
  )

pso_agd <- pso_agd \%>\%
  mutate(
    # Variable transformations
    bsa_mean = bsa_mean / 100,
    bsa_sd = bsa_sd / 100,
    prevsys = prevsys / 100,
    psa = psa / 100,
    weight_mean = weight_mean / 10,
    weight_sd = weight_sd / 10,
    durnpso_mean = durnpso_mean / 10,
    durnpso_sd = durnpso_sd / 10,
    # Treatment classes
    trtclass = case_when(trtn == 1 ~ "Placebo",
                         trtn \%in\% c(2, 3, 5, 6) ~ "IL blocker",
                         trtn == 4 ~ "TNFa blocker")
  )

# Exclude small number of individuals with missing covariates
pso_ipd <- filter(pso_ipd, complete)

pso_net <- combine_network(
  set_ipd(pso_ipd,
          study = studyc,
          trt = trtc,
          r = pasi75,
          trt_class = trtclass),
  set_agd_arm(pso_agd,
              study = studyc,
              trt = trtc,
              r = pasi75_r,
              n = pasi75_n,
              trt_class = trtclass)
)

# Print network details
pso_net

# Add integration points to the network
pso_net <- add_integration(pso_net,
  durnpso = distr(qgamma, mean = durnpso_mean, sd = durnpso_sd),
  prevsys = distr(qbern, prob = prevsys),
  bsa = distr(qlogitnorm, mean = bsa_mean, sd = bsa_sd),
  weight = distr(qgamma, mean = weight_mean, sd = weight_sd),
  psa = distr(qbern, prob = psa),
  n_int = 64)


## Adding integration points to a data frame, e.g. for prediction
# Define a data frame of covariate summaries
new_agd_int <- data.frame(
  bsa_mean = 0.6,
  bsa_sd = 0.3,
  prevsys = 0.1,
  psa = 0.2,
  weight_mean = 10,
  weight_sd = 1,
  durnpso_mean = 3,
  durnpso_sd = 1)

# Adding integration points, using the weighted average correlation matrix
# computed for the plaque psoriasis network
new_agd_int <- add_integration(new_agd_int,
  durnpso = distr(qgamma, mean = durnpso_mean, sd = durnpso_sd),
  prevsys = distr(qbern, prob = prevsys),
  bsa = distr(qlogitnorm, mean = bsa_mean, sd = bsa_sd),
  weight = distr(qgamma, mean = weight_mean, sd = weight_sd),
  psa = distr(qbern, prob = psa),
  cor = pso_net$int_cor,
  n_int = 64)

# Here, since we reused the correlation matrix pso_net$int_cor from the
# network, the correct setting of cor_adjust = "spearman" is automatically
# applied

new_agd_int

}
\references{
\insertAllCited{}
}
